#!/usr/bin/env python
# -*- coding: latin-1 -*-
# install wxpython on RaspberryPi using: sudo apt install wxpython-tools
# for windows use: pip install wxpython
################################################################
##                                                            ##
##                          Version 0.8.5                     ##
##                                                            ##
################################################################
# v.0.1: first wxPython version of Les Wright's PySpectrometer2, now called PySpectrometer3, added various functions like adjustable capture-width, filtering, gain and exposure, and FWHM measuring tool
# v.0.2: unsuccessful attempt implementing double buffering based on code by Virgil Stokes: https://wiki.wxpython.org/DoubleBufferedDrawing
# v.0.3: successful implementation of double buffering based on code by Jean-Michel Fauth, Switzerland: https://discuss.wxpython.org/t/drawing-with-a-buffered-bitmap-a-complete-sample-code/7090
# v.0.4: moved Fauth's double buffering drawing function DoSomeDrawing to main class
# v.0.5: converted processFrame to double buffering, implemented new save function
# v.0.6: removed DoSomeDrawing, added taking darks
# V.0.7: Added detection of spectrum in vertical direction, Saving and Reading Preferences. Improved taking Darks and graph annotation.
# v.0.8: Removed all "== True" and "== False" statements, added Argon/Neon sequence, replaced readCal by ReadCalibrationFile, split AutoCalibrate into AutoCalibrate and CalculateCalibrationCoefficients, added chi-squared calculation
# v.0.8.1: replaced grid drawing function by one that can handle string 3rd degree polynomes, removed use or original graticuleData, tens and fifties. Implemented and tested 12bit mono cameras (16bit prepared, but not tested).
# v.0.8.2: Added full camera preview, FITS export. Simplyfied Save-function.
# v.0.8.3: Started implementation of Waterfall display, needs more work.
# v.0.8.4: Added plots of lamp graphs. Calculating chi-squared based on residuals, removing outliers using MAD-filter. Added polyfit line to residualplot. peakThreshold is set in StartCam() as 1% of bitdepth.
# v.0.8.5: Added button and keyboard shortcut to reset the temporal running average filter. Added counter to "Running Avg: xxx", e.g.: "Running Avg: 35/200". Added stability indication.


import wx
import wx.grid as gridlib
import wx.lib.plot as plot
#import matplotlib.pyplot as plotlib
import threading
import cv2
import numpy as np
import random
#from specFunctions import wavelength_to_rgb,savitzky_golay,peakIndexes,readcal,writecal,background,generateGraticule
from specFunctions import wavelength_to_rgb,savitzky_golay,peakIndexes
import platform
import time
import math
from pathlib import Path
import warnings
from scipy.optimize import curve_fit
from PyAstronomy import pyasl as pyastro # requires astropy to be installed as well!
from astropy.io import fits
from numpy import loadtxt # for lamp graph
import scipy.interpolate as spi # for lamp graph
from scipy import stats # for stability measurement

if platform.system()== "Windows":
    platformIsWindows = True
    import zwoasi as asi
else:
    platformIsWindows = False
    from picamera2 import Picamera2

PROGRAMNAME = "PySpectrometer3"
PROGRAMVERSION = "0.8.5"

# definition of clibration lamps
# [["Name of bulb",
#   [Peak wavelengths in nm, comma separated],
#   Default First peak index to be used,
#   Default Last peak index to be used,
#   Detection level of first and last peak as percentage of maximum intensity,
#   Try Gaussian fit during calibration [True/False],
#   "path to corresponding example graph",
#   "Author, 'title of publication', year of publication",
#   "url to publication"
#   "comment"
CALIBRATIONLAMPS = [["Fluorescent",
     [401.169, 404.656, 435.833, 487.7, 542.4, 546.074, 577.7, 580.2, 584.0, 587.6, 593.4, 599.7, 611.6, 625.7, 631.1, 650.8, 662.6, 687.7, 693.7, 707, 709, 712.3, 758.932, 763.511, 811.531],
     1,
     11,
     0.05,
     False,
     "./lampgraphs/fluorescent.csv",
     "Deglr6328, 'File:Fluorescent lighting spectrum peaks labelled.gif',2005",
     "https://en.wikipedia.org/wiki/File:Fluorescent_lighting_spectrum_peaks_labelled.gif"
     "Mix of measured wavelength peaks by Deglr6328 and the well known actual line locations. First peak from BASS, element Europium (Eu: 401.169nm)"],
     
     ["Philips S10 NeXe",
#     [421.372, 460.0, 461.8, 466.9, 484.329, 491.651, 540.100, 585.248, 627.082, 640.225, 651.283, 667.828, 692.947, 703.241, 711.96, 724.517, 743.89, 748.89, 753.58, 758.47, 764.20],
     [421.372, 433.052, 446.219, 458.28, 462.43, 467.12, 469.70, 480.70, 482.97, 484.33, 491.65, 492.32, 502.83, 503.78, 508.04, 511.65, 514.50, 520.39, 533.08, 534.11, 534.33, 540.06, 585.25, 588.19, 594.48, 597.55, 603.00, 607.43, 609.62, 614.31, 616.36, 621.73, 626.65, 630.48, 633.44, 638.30, 640.23, 650.65, 653.29, 659.89, 667.83, 671.70, 672.80, 682.73, 688.22, 692.95, 703.24, 711.96, 717.39, 724.52, 743.89, 748.89, 753.58, 758.47, 764.20],
     22,
     46,
     0.1,
     True,
     "./lampgraphs/NeXe_Philips_S10.csv",
     "Richard Walker, 'Multi Spektral-Kalibrierlampe mit modifizierten Glimmstartern', 2021",
     "https://www.ursusmajor.ch/downloads/multispektrallampe.pdf",
     ""],
     
     ["Shelyak S0148 NeAr",
     [394.9, 404.44, 419.10, 420.067, 427.40, 433.36, 451.073, 503.78, 520.39, 534.39, 540.06, 576.44, 585.249, 594.48, 603.00, 607.43, 609.62, 614.306, 621.73, 626.65, 630.48, 633.44, 640.225, 650.65, 659.90, 667.728, 671.70, 675.28, 687.13, 696.543, 703.24, 706.722, 714.70, 727.294, 738.40, 750.387, 763.511, 772.38, 794.32, 794.82, 800.62, 801.48, 811.53, 813.64],
     10,
     35,
     0.1,
     True,
     "./lampgraphs/NeAr_Shelyak.csv",
     "Oriel Instruments, 'Typical Spectra of Spectral Calib Lamps', n.d., Shelyak, Alpy calibration User Guide, 2014",
     "https://www.newport.com/medias/sys_master/images/images/h55/hfd/8797293281310/Typical-Spectra-of-Spectral-Calib-Lamps.pdf",
     ""],
     
     ["Relco-SC480 NeAr",
     [394.61, 407.20, 413.172, 415.859, 420.067, 427.753, 430.01, 442.600, 451.073, 454.505, 460.957, 465.79, 476.487, 480.602, 487.986, 496.508, 518.774, 540.056, 549.587, 555.870, 560.673, 576.441, 607.434, 609.616, 614.306, 616.359, 621.728, 626.649, 630.479, 633.443, 638.299, 640.225, 650.653, 653.288, 656.285, 659.895, 667.728, 671.7043, 675.283, 687.129, 692.947, 696.543, 706.722, 714.704, 738.398, 763.511],
     4,
     45,
     0.3,
     True,
     "./lampgraphs/NeArHe_Relco_SC480.csv",
     "Richard Walker, 'Glow Starter RELCO SC480: Atlas of Emission Lines', 2017",
     "https://www.ursusmajor.ch/downloads/sques-relco-sc480-calibration-lines-5.0.pdf",
     ""]
     ]

# Enabling/disabling output to terminal
VERBOSE = False

# toolbar code constants
ID_SAV = wx.ID_HIGHEST +  1
ID_CAL = wx.ID_HIGHEST +  2
ID_DAR = wx.ID_HIGHEST +  3
ID_NOR = wx.ID_HIGHEST +  4
ID_FPL = wx.ID_HIGHEST +  5
ID_FMI = wx.ID_HIGHEST +  6
ID_GPL = wx.ID_HIGHEST +  7
ID_GMI = wx.ID_HIGHEST +  8
ID_EPL = wx.ID_HIGHEST +  9
ID_EMI = wx.ID_HIGHEST + 10
ID_CPL = wx.ID_HIGHEST + 11
ID_CMI = wx.ID_HIGHEST + 12
ID_MEG = wx.ID_HIGHEST + 13
ID_MEB = wx.ID_HIGHEST + 14
ID_SGT = wx.ID_HIGHEST + 15
ID_SGP = wx.ID_HIGHEST + 16
ID_HPE = wx.ID_HIGHEST + 17
ID_SCL = wx.ID_HIGHEST + 18
ID_SRP = wx.ID_HIGHEST + 19
ID_CVE = wx.ID_HIGHEST + 20
ID_CLA = wx.ID_HIGHEST + 21
ID_SCP = wx.ID_HIGHEST + 22
ID_RES = wx.ID_HIGHEST + 23

# shortcut keys
ID_SAV_KEY = "S" # Save
ID_CAL_KEY = "C" # Calibrate
ID_DAR_KEY = "-" # Darks
ID_NOR_KEY = "N" # Normalise
ID_FPL_KEY = "R" # Running average filter up
ID_FMI_KEY = "F" # Running average filter down
ID_GPL_KEY = "G" # Gain up
ID_GMI_KEY = "B" # Gain down
ID_EPL_KEY = "E" # Exposure up
ID_EMI_KEY = "D" # Exposure down
ID_CPL_KEY = "Q" # Crop height up
ID_CMI_KEY = "A" # Crop height down
ID_MEG_KEY = "M" # Measure Gaussian FWHM
ID_MEB_KEY = "[" # Measure Block FWHM
ID_SGT_KEY = "T" # SavGol filter on/off
ID_SGP_KEY = "Y" # SavGol filter up
ID_HPE_KEY = "P" # Holdpeaks
ID_SCL_KEY = "L" # Show calibration lines
ID_SRP_KEY = "O" # Show residual plot
ID_CVE_KEY = "V" # Vertically centre the spectrum
ID_CLA_KEY = "=" # Configure the calibration lamp
ID_SCP_KEY = "Z" # Show full camera preview
ID_RES_KEY = "X" # Reset running average (restarts sampling from the last frame)


CURSOR_COLOUR_NORMAL = (255,0,0)
CURSOR_COLOUR_FWHM_GAUSSIAN = (0,255,0)
CURSOR_COLOUR_FWHM_BLOCK = (0,0,255)

FWHM_GAUSSIAN = 0
FWHM_BLOCK = 1

POLYFITS = ["1st Degree","2nd Degree","3rd Degree","4th Degree"]

# ========================================================================================
class PySpectrometer3(wx.Frame):


    
    # ------------------------------------------------------------------------------------
    def __init__(self, parent):
        default = wx.MINIMIZE_BOX | wx.MAXIMIZE_BOX | wx.RESIZE_BORDER | wx.SYSTEM_MENU | wx.CAPTION | wx.CLOSE_BOX | wx.CLIP_CHILDREN | wx.NO_FULL_REPAINT_ON_RESIZE
        super().__init__(parent, style = default)


        self.programName = PROGRAMNAME+" "+PROGRAMVERSION
        self.thePrefFileName = "preferences.txt"

        self.calibrationLampId = 0
        self.calibrationRejectionLevel = 2 # nm
        self.calibrationSpectrumQuality = CALIBRATIONLAMPS[self.calibrationLampId][5]
        self.CalibrationLampSetUpFrameIsOpen = False
        self.calibrationFirstPeak = CALIBRATIONLAMPS[self.calibrationLampId][2]
        self.calibrationLastPeak = CALIBRATIONLAMPS[self.calibrationLampId][3]
        self.calibrationDetectionLevel = CALIBRATIONLAMPS[self.calibrationLampId][4]
        self.polyfitDegree = 2
        self.calibrationResidualPlotData = []
        self.calibrationResidualPlotDataLinear = []
        self.calibrationPolynomeDifferenceWithLinear = []
        self.calibrationResidualPlotFrameIsOpen = False
        self.calibrationRequested = False
        self.dispersion = 0 # nm/px
        self.R_sq = 1
        self.chiSquaredPdoF = 1

        self.captureStarted = False
        self.intensity = []
        self.intensityFloat = []
        self.intensityMax = 0
        self.resetRunningAverageData = True
        self.resetRunningAverageStDev = False
        self.runningAverageStDev = 0
        self.runningAverageStDevAmount = 30 # the fixed amount of frames over which the dStDev is averaged
        self.runningAverageStDevRegressionArrayNumbers = list(range(self.runningAverageStDevAmount))
        self.runningAverageStDevRegressionArrayLastValues = [0] * self.runningAverageStDevAmount
        self.runningAverageStDevRegressionCriterion = 0 # as soon as the slope is below this value the system is presumed stable, is updated every frame as it depends on settings
        self.runningAvgAmount = 30 # amount of frames for the temporal running average, can be altered during program execution
        self.runningAvgCounter = 0 # counter for display purposes only
        self.normaliseData = []
        self.darkData = []
        self.waterfall = [] #blank image for Waterfall
        self.graphFillHeight = 0.90 # percentage of graph to be filled 1.00 = 100%
        self.cropHeight = 40 # the height of the video capture to be used
        self.sampleHeight = 15 # the number of sample-rows to be taken from the cropHeight to populate the spectrum
        self.sampleHeightMax = 41
        self.verticalOffset = 0 # offset from centre of the maximum intensity of the spectrum within the whole image
        self.dispWaterfall = False
        self.showNormalised = False
        self.applyDarks = False
        self.isCalibrated = False
        self.originalCalibrationPixels = []
        self.updatedCalibrationPixels = []
        self.wavelengthData = []
        self.correspondingWavelengths = []
        self.correspondingWavelengthDifferences = []
        self.calibrationStartTime = time.time()
        self.maxCalibrationShowTime = 10 # seconds, the amount of time that the calibration lines remain visible after start-up and calibration
        self.cursorColour = CURSOR_COLOUR_NORMAL
        self.doFWHM_Measurement = False
        self.doGaussianFWHM_Measurement = False
        self.FWHM_MeasurementLeftDone = False
        self.FWHM_MeasurementRightDone = False
        self.FWHM_RangeStart = 0
        self.FWHM_RangeEnd = 0
        self.FWHM_PeakPosLeft = 0
        self.FWHM_PeakPosRight = 0
        self.FWHM_DataFrameIsOpen = False
        #settings for peak detect
        self.useSavitzkyGolayFilter = False
        self.savgolFilterPolynomial = 7 #savgol filter polynomial max val 15
        self.peakMinDist = 20 #minumum distance between peaks max val 100
        self.peakThreshold = 0 #peakThreshold is set in StartCam()
        self.holdPeaks = False
        self.cameraName = "-"
        self.cameraBitDepth = 8 # the bit depth of the video stream
        self.cameraIsMono = True
        self.cameraWidth = 0
        self.cameraHeight = 0
        self.cameraGain = 0
        self.cameraMaxGain = 0
        self.cameraExposure = 0
        self.cameraMaxExposure = 0
        self.cameraIsPi = False
        self.cameraIsUSB = False
        self.cameraIsASI = False
        self.canSetGain = False
        self.canSetExposure = False
        self.cameraPreviewFrameIsOpen = False

        self.BufferBmp = None

        self.windowPosX = 100
        self.windowPosY = 100

        sw,sh = wx.DisplaySize()
        self.windowSizeX = int(sw*0.75)
        self.windowSizeY = int(sh*0.75)
        self.windowIsMaximized = False

        self.ReadPrefs()
        self.InitUI()
#        self.Centre()
    


    # ------------------------------------------------------------------------------------
    def InitUI(self):

        menubar = wx.MenuBar()

        self.count = 0
        self.dataIdx = -1
        self.SetSize((self.windowSizeX, self.windowSizeY))
        self.shift_down = False

        # ------ View Menu ------
        viewMenu = wx.Menu()
        self.shst = viewMenu.Append(wx.ID_ANY, 'Show statusbar',
            'Show Statusbar', kind=wx.ITEM_CHECK)
        self.shtl = viewMenu.Append(wx.ID_ANY, 'Show toolbar',
            'Show Toolbar', kind=wx.ITEM_CHECK)

        viewMenu.Check(self.shst.GetId(), True)
        viewMenu.Check(self.shtl.GetId(), True)

        self.Bind(wx.EVT_MENU, self.ToggleStatusBar, self.shst)
        self.Bind(wx.EVT_MENU, self.ToggleToolBar, self.shtl)

        menubar.Append(viewMenu, '&View')
        
        # ------ Calibration Menu ------
        self.calibrationMenu = wx.Menu()
        self.cm = [0] * len(CALIBRATIONLAMPS[:])
        
        idx = 0
        for lamp in CALIBRATIONLAMPS:
            self.cm[idx] = self.calibrationMenu.Append(wx.ID_ANY, lamp[0], lamp[0], kind=wx.ITEM_CHECK)
            self.calibrationMenu.Check(self.cm[idx].GetId(), self.calibrationLampId == idx)
            self.Bind(wx.EVT_MENU, lambda event, arg=idx: self.ToggleCalibrationLamp(arg), self.cm[idx])
            idx += 1
        self.calibrationMenu.AppendSeparator()
        
        self.cmsu = self.calibrationMenu.Append(wx.ID_ANY, 'Configure lamp ['+ID_CLA_KEY+']', 'Configure lamp')
        self.Bind(wx.EVT_MENU, self.showCalibrationLampSetUpFrame, self.cmsu)

        menubar.Append(self.calibrationMenu, '&Calibration lamp')

        # ------ Polyfit Menu ------
        self.polyfitMenu = wx.Menu()
        self.po1st = self.polyfitMenu.Append(wx.ID_ANY, '1st degree', '1st degree', kind=wx.ITEM_CHECK)
        self.po2nd = self.polyfitMenu.Append(wx.ID_ANY, '2nd degree', '2nd degree', kind=wx.ITEM_CHECK)
        self.po3rd = self.polyfitMenu.Append(wx.ID_ANY, '3rd degree', '3rd degree', kind=wx.ITEM_CHECK)
        self.po4th = self.polyfitMenu.Append(wx.ID_ANY, '4th degree', '4th degree', kind=wx.ITEM_CHECK)

        self.polyfitMenu.Check(self.po1st.GetId(), self.polyfitDegree == 1)
        self.polyfitMenu.Check(self.po2nd.GetId(), self.polyfitDegree == 2)
        self.polyfitMenu.Check(self.po3rd.GetId(), self.polyfitDegree == 3)
        self.polyfitMenu.Check(self.po4th.GetId(), self.polyfitDegree == 4)

        self.Bind(wx.EVT_MENU, lambda event, arg=1: self.TogglePolyFit(arg), self.po1st)
        self.Bind(wx.EVT_MENU, lambda event, arg=2: self.TogglePolyFit(arg), self.po2nd)
        self.Bind(wx.EVT_MENU, lambda event, arg=3: self.TogglePolyFit(arg), self.po3rd)
        self.Bind(wx.EVT_MENU, lambda event, arg=4: self.TogglePolyFit(arg), self.po4th)

        menubar.Append(self.polyfitMenu, '&Polyfit')
        
        self.SetMenuBar(menubar)

        # ------ Toolbar ------
        self.toolbar = self.CreateToolBar()
        tool_sav = self.toolbar.AddTool(ID_SAV, 'Save', wx.Bitmap('icons/icon_save.png'), "Save the spectrum ["+ID_SAV_KEY+"]")
        tool_cve = self.toolbar.AddTool(ID_CVE, 'Centre vertical', wx.Bitmap('icons/icon_centre_spectrum.png'), "Centre the spectrum vertically ["+ID_CVE_KEY+"]")
        tool_cal = self.toolbar.AddTool(ID_CAL, 'Calibrate', wx.Bitmap('icons/icon_calibrate.png'), "Calibrate the spectrum using known wavelengths ["+ID_CAL_KEY+"]")
        tool_dar = self.toolbar.AddTool(ID_DAR, 'Dark', wx.Bitmap('icons/icon_dark.png'), "Take darks to correct for hot pixels ["+ID_DAR_KEY+"]")
        tool_nor = self.toolbar.AddTool(ID_NOR, 'Normalise', wx.Bitmap('icons/icon_normalise.png'), "Normalise the spectrum ["+ID_NOR_KEY+"]")
        tool_cpl = self.toolbar.AddTool(ID_CPL, 'Crop plus', wx.Bitmap('icons/icon_crop_widen.png'), "Widen sample height by 1 row (decreases speed and noise)["+ID_CPL_KEY+"]")
        tool_cmi = self.toolbar.AddTool(ID_CMI, 'Crop min', wx.Bitmap('icons/icon_crop_reduce.png'), "Reduce sample height by 1 row (increases speed and noise) ["+ID_CMI_KEY+"]")
        tool_fpl = self.toolbar.AddTool(ID_FPL, 'Filter plus', wx.Bitmap('icons/icon_filter_plus.png'), "Increase running average filter by 1 exposure (shift-click by 10) ["+ID_FPL_KEY+"]")
        tool_fmi = self.toolbar.AddTool(ID_FMI, 'Filter min', wx.Bitmap('icons/icon_filter_min.png'), "Decrease running average filter by 1 exposure (shift-click by 10) ["+ID_FMI_KEY+"]")
        tool_res = self.toolbar.AddTool(ID_RES, 'Reset running average', wx.Bitmap('icons/icon_filter_reset.png'), "Reset running average ["+ID_RES_KEY+"]")
        if self.useSavitzkyGolayFilter:
            iconStr = 'icons/icon_SG_filter_on.png'
        else:
            iconStr = 'icons/icon_SG_filter_off.png'
        tool_sgt = self.toolbar.AddTool(ID_SGT, 'Savitzky-Golay ON/OFF', wx.Bitmap(iconStr), "Toggle Savitzky-Golay filter ON/OFF ["+ID_SGT_KEY+"]")
        tool_sgp = self.toolbar.AddTool(ID_SGP, 'Increase Savitzky-Golay', wx.Bitmap('icons/icon_SG_filter_plus.png'), "Increase Savitzky-Golay (>15 returns to 0) ["+ID_SGP_KEY+"]")
        if self.holdPeaks:
            iconStr = 'icons/icon_holdpeaks_active.png'
        else:
            iconStr = 'icons/icon_holdpeaks.png'
        tool_hpe = self.toolbar.AddTool(ID_HPE, 'Hold peaks', wx.Bitmap(iconStr), "Toggle hold peaks ON/OFF ["+ID_HPE_KEY+"]")
        tool_gmi = self.toolbar.AddTool(ID_GMI, 'Gain min', wx.Bitmap('icons/icon_gain_min2.png'), "Decrease gain by 1 (shift-click by 10) ["+ID_GMI_KEY+"]")
        tool_gpl = self.toolbar.AddTool(ID_GPL, 'Gain plus', wx.Bitmap('icons/icon_gain_plus2.png'), "Increase gainby 1 (shift-click by 10) ["+ID_GPL_KEY+"]")
        tool_emi = self.toolbar.AddTool(ID_EMI, 'Exposure min', wx.Bitmap('icons/icon_exposure_min.png'), "Decrease exposure by 1% (shift-click by 10%) ["+ID_EMI_KEY+"]")
        tool_epl = self.toolbar.AddTool(ID_EPL, 'Exposure plus', wx.Bitmap('icons/icon_exposure_plus.png'), "Increase exposure by 1% (shift-click by 10%) ["+ID_EPL_KEY+"]")
        tool_meg = self.toolbar.AddTool(ID_MEG, 'Measure Gaussian FWHM', wx.Bitmap('icons/icon_GaussianFWHM.png'), "Measure the FWHM of a Gaussian peak ["+ID_MEG_KEY+"]")
        tool_meb = self.toolbar.AddTool(ID_MEB, 'Measure Block FWHM', wx.Bitmap('icons/icon_BlockFWHM.png'), "Measure the FWHM of a block peak ["+ID_MEB_KEY+"]")
        tool_scl = self.toolbar.AddTool(ID_SCL, 'Show calibration lines', wx.Bitmap('icons/icon_showCalLines.png'), "Show the calibration lines ["+ID_SCL_KEY+"]")
        tool_srp = self.toolbar.AddTool(ID_SRP, 'Show residual plot', wx.Bitmap('icons/icon_showResiduals.png'), "Show residual plot ["+ID_SRP_KEY+"]")
        tool_scp = self.toolbar.AddTool(ID_SCP, 'Full camera preview', wx.Bitmap('icons/icon_preview.png'), "Show full camera preview ["+ID_SCP_KEY+"]")

        self.toolbar.EnableTool(ID_SAV, False)
        self.toolbar.EnableTool(ID_CAL, False)
        self.toolbar.EnableTool(ID_DAR, False)
        self.toolbar.EnableTool(ID_NOR, False)
        self.toolbar.EnableTool(ID_CMI, False)
        self.toolbar.EnableTool(ID_CPL, False)
        self.toolbar.EnableTool(ID_FMI, False)
        self.toolbar.EnableTool(ID_FPL, False)
        self.toolbar.EnableTool(ID_GMI, False)
        self.toolbar.EnableTool(ID_GPL, False)
        self.toolbar.EnableTool(ID_EMI, False)
        self.toolbar.EnableTool(ID_EPL, False)
        self.toolbar.EnableTool(ID_MEG, False)
        self.toolbar.EnableTool(ID_MEB, False)
        self.toolbar.EnableTool(ID_SGT, False)
        self.toolbar.EnableTool(ID_SGP, False)
        self.toolbar.EnableTool(ID_HPE, False)
        self.toolbar.EnableTool(ID_SCL, False)
        self.toolbar.EnableTool(ID_SRP, False)
        self.toolbar.EnableTool(ID_CVE, False)
        self.toolbar.EnableTool(ID_SCP, False)
        self.toolbar.EnableTool(ID_RES, False)
        self.toolbar.Realize()
        self.Bind(wx.EVT_TOOL, self.Save, tool_sav)
        self.Bind(wx.EVT_TOOL, self.RecentreSpectrum, tool_cve)
        self.Bind(wx.EVT_TOOL, self.Calibrate, tool_cal)
        self.Bind(wx.EVT_TOOL, self.TakeDarks, tool_dar)
        self.Bind(wx.EVT_TOOL, self.ToggleNormalisation, tool_nor)
        self.Bind(wx.EVT_TOOL, lambda event, arg=-1: self.CropControl(event, arg), tool_cmi)
        self.Bind(wx.EVT_TOOL, lambda event, arg=1: self.CropControl(event, arg), tool_cpl)
        self.Bind(wx.EVT_TOOL, lambda event, arg=-1: self.FilterControl(event, arg), tool_fmi)
        self.Bind(wx.EVT_TOOL, lambda event, arg=1: self.FilterControl(event, arg), tool_fpl)
        self.Bind(wx.EVT_TOOL, lambda event, arg=-1: self.GainControl(event, arg), tool_gmi)
        self.Bind(wx.EVT_TOOL, lambda event, arg=1: self.GainControl(event, arg), tool_gpl)
        self.Bind(wx.EVT_TOOL, lambda event, arg=-1: self.ExposureControl(event, arg), tool_emi)
        self.Bind(wx.EVT_TOOL, lambda event, arg=1: self.ExposureControl(event, arg), tool_epl)
        self.Bind(wx.EVT_TOOL, lambda event, arg=0: self.ToggleSavitzkyGolayFilter(event, arg), tool_sgt)
        self.Bind(wx.EVT_TOOL, lambda event, arg=1: self.ToggleSavitzkyGolayFilter(event, arg), tool_sgp)
        self.Bind(wx.EVT_TOOL, lambda event, arg=FWHM_GAUSSIAN: self.ToggleMeasureFWHM(event, arg), tool_meg)
        self.Bind(wx.EVT_TOOL, lambda event, arg=FWHM_BLOCK: self.ToggleMeasureFWHM(event, arg), tool_meb)
#        self.Bind(wx.EVT_TOOL, self.ToggleMeasureFWHM, tool_mea)
        self.Bind(wx.EVT_TOOL, self.ToggleHoldPeaks, tool_hpe)
        self.Bind(wx.EVT_TOOL, self.ShowCalibrationLines, tool_scl)
        self.Bind(wx.EVT_TOOL, self.ToggleResidualPlot, tool_srp)
        self.Bind(wx.EVT_TOOL, self.ToggleCameraPreviewFrame, tool_scp)
        self.Bind(wx.EVT_TOOL, self.ResetRunningAverage, tool_res)
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
        self.Bind(wx.EVT_MAXIMIZE, self.OnMaximize)
        self.Bind(wx.EVT_ICONIZE, self.OnMinimize)
        
        
        self.statusbar = self.CreateStatusBar()
        self.statusbar.SetStatusText('Ready')

        self.panel = wx.Panel(self)
        self.panel.SetBackgroundColour("gray")
        vbox = wx.BoxSizer(wx.VERTICAL)

        self.midPanTop = InfoPanel(self.panel, self.statusbar, (0, 40), "info")
        self.midPanCentre = ImagePanel(self.panel, self.statusbar, (0, 40), "capture")
        self.midPanBottom = GraphWindow(self.panel, -1, self.statusbar, (0, 0), "graph")

        self.midPanBottom.Bind(wx.EVT_LEFT_DOWN, self.SetCursorPosition)
        self.midPanBottom.Bind(wx.EVT_LEFT_UP, self.GetDataIdx)
        self.midPanBottom.Bind(wx.EVT_MOTION, self.OnMouseMove)
        self.midPanBottom.Bind(wx.EVT_SIZE, self.OnSize)

        self.Bind(wx.EVT_CHAR_HOOK, self.OnKeyPress)

        self.timer = wx.Timer(self, self.midPanBottom.ID_TIMER)
        self.Bind(wx.EVT_TIMER, self.OnTimer, id=self.midPanBottom.ID_TIMER)

        vbox.Add(self.midPanTop, 0, wx.EXPAND | wx.TOP, 1)
        vbox.Add(self.midPanCentre, 0, wx.EXPAND | wx.TOP, 1)
        vbox.Add(self.midPanBottom, 5, wx.EXPAND | wx.TOP | wx.BOTTOM, 1)
        self.panel.SetSizer(vbox)

        self.SetTitle(self.programName)
#        self.Centre()
        self.SetPosition((self.windowPosX,self.windowPosY))
        self.SetMinSize((500,300))
        
        self.Bind(wx.EVT_CLOSE, self.OnExit)
        self.midPanTop.runningAvgLabel.SetLabel("Running AVG: {}/{}".format(self.runningAvgCounter,self.runningAvgAmount))
        self.midPanTop.sampleHeightLabel.SetLabel("Sample height: "+str(self.sampleHeight)+" rows")

    # ------------------------------------------------------------------------------------
    def OnSize(self, event):
        # Get the size of the drawing area in pixels.
        self.windowWidth, self.windowHeight = self.midPanBottom.GetSize()
        self.midPanBottom.windowWidth = self.windowWidth
        self.midPanBottom.windowHeight = self.windowHeight
        # Create BufferBmp and set the same size as the drawing area.
        self.BufferBmp = wx.Bitmap(self.windowWidth, self.windowHeight)
        self.midPanBottom.BufferBmp = self.BufferBmp

    # ------------------------------------------------------------------------------------
    def OnMaximize(self, evt):
        self.windowIsMaximized = True
        print("maximizing",self.GetSize(),wx.DisplaySize())
        
    # ------------------------------------------------------------------------------------
    def OnMinimize(self, evt):
        self.windowIsMaximized = False
        print("minimizing",self.GetSize())
        
    # ------------------------------------------------------------------------------------
    def OnEraseBackground(self, evt):
        #print("Erasing background")
        # do nothing
        pass
        

    # ------------------------------------------------------------------------------------
    def GetCalData(self):

        #Go grab the computed calibration data
        caldata = self.ReadCalibrationFile()

        if self.isCalibrated:
            self.SetCalibrationStatus()

    # ------------------------------------------------------------------------------------
    def enableToolbar(self):
        if self.canSetGain:
            self.toolbar.EnableTool(ID_GMI, True)
            self.toolbar.EnableTool(ID_GPL, True)
        if self.canSetExposure:
            self.toolbar.EnableTool(ID_EMI, True)
            self.toolbar.EnableTool(ID_EPL, True)

    # ------------------------------------------------------------------------------------
    def ToggleToolBar(self, e):

        if self.shtl.IsChecked():
            self.toolbar.Show()
        else:
            self.toolbar.Hide()
        self.SetSize(self.GetSize()-(1,1))
        self.SetSize(self.GetSize()+(1,1))

    # ------------------------------------------------------------------------------------
    def ToggleStatusBar(self, e):

        if self.shst.IsChecked():
            self.statusbar.Show()
        else:
            self.statusbar.Hide()
        # now resizing the frame to invoke an update as simply calling update_bitmap() will not work due to not update self.GetSize()
        self.SetSize(self.GetSize()-(1,1))
        self.SetSize(self.GetSize()+(1,1))

    # ------------------------------------------------------------------------------------
    def ToggleCalibrationLamp(self, lampId):

        idx = 0
        for lamp in CALIBRATIONLAMPS:
            if idx != lampId:
                self.calibrationMenu.Check(self.cm[idx].GetId(), False)
            else:
                self.calibrationMenu.Check(self.cm[idx].GetId(), True) # this is done to avoid that the current calibration lamp can be deselected
            idx += 1
        self.calibrationLampId = lampId
        self.calibrationFirstPeak = CALIBRATIONLAMPS[self.calibrationLampId][2]
        self.calibrationLastPeak = CALIBRATIONLAMPS[self.calibrationLampId][3]
        self.calibrationDetectionLevel = CALIBRATIONLAMPS[self.calibrationLampId][4]
        self.calibrationSpectrumQuality = CALIBRATIONLAMPS[self.calibrationLampId][5]
        if self.CalibrationLampSetUpFrameIsOpen:
            self.showCalibrationLampSetUpFrame(None) # close it
            self.showCalibrationLampSetUpFrame(None) # re-open it
        if self.calibrationResidualPlotFrameIsOpen:
            self.ToggleResidualPlot(None) # close it
            self.ToggleResidualPlot(None) # re-open it
    # ------------------------------------------------------------------------------------
    def ToggleMeasureFWHM(self, evt, theMethod):
        # here we only toggle the FWHM function on/off, registring the mouse-click locations is done in self.GetDataIdx(), calculating the FWHM in self.calculateFWHM()
        # first check if there already is a FWHM measurement active
        if self.doFWHM_Measurement:
            self.FWHM_MeasurementLeftDone = False
            self.FWHM_MeasurementRightDone = False
            if theMethod == FWHM_GAUSSIAN and self.doGaussianFWHM_Measurement:
                self.doFWHM_Measurement = False
                self.doGaussianFWHM_Measurement = False
            elif theMethod == FWHM_GAUSSIAN and not self.doGaussianFWHM_Measurement:
                self.doGaussianFWHM_Measurement = True
            elif theMethod == FWHM_BLOCK and self.doGaussianFWHM_Measurement:
                self.doGaussianFWHM_Measurement = False
            elif theMethod == FWHM_BLOCK and not self.doGaussianFWHM_Measurement:
                self.doFWHM_Measurement = False
                self.doGaussianFWHM_Measurement = False
        else:
            self.doFWHM_Measurement = True
            if theMethod == FWHM_GAUSSIAN:
                self.doGaussianFWHM_Measurement = True
            
        if not self.doFWHM_Measurement:
            self.toolbar.SetToolNormalBitmap(ID_MEG, wx.Bitmap('icons/icon_GaussianFWHM.png'))
            self.toolbar.SetToolNormalBitmap(ID_MEB, wx.Bitmap('icons/icon_BlockFWHM.png'))
            self.SetCalibrationStatus()
            if self.FWHM_DataFrameIsOpen:
                self.FWHM_DataFrame.Close()
        else:
            if self.doGaussianFWHM_Measurement:
                self.toolbar.SetToolNormalBitmap(ID_MEG, wx.Bitmap('icons/icon_GaussianFWHM_active.png'))
                self.toolbar.SetToolNormalBitmap(ID_MEB, wx.Bitmap('icons/icon_BlockFWHM.png'))
            else:
                self.toolbar.SetToolNormalBitmap(ID_MEG, wx.Bitmap('icons/icon_GaussianFWHM.png'))
                self.toolbar.SetToolNormalBitmap(ID_MEB, wx.Bitmap('icons/icon_BlockFWHM_active.png'))

            if not self.FWHM_DataFrameIsOpen:
                self.FWHM_DataFrame = FWHM_resultsFrame(self.panel)
                self.FWHM_DataFrameIsOpen = True
            else:
                self.FWHM_DataFrame.Close()
                self.FWHM_DataFrame = FWHM_resultsFrame(self.panel)
                self.FWHM_DataFrameIsOpen = True    

    # ------------------------------------------------------------------------------------
    def TogglePolyFit(self, degree):

        self.polyfitMenu.Check(self.po1st.GetId(), degree == 1)
        self.polyfitMenu.Check(self.po2nd.GetId(), degree == 2)
        self.polyfitMenu.Check(self.po3rd.GetId(), degree == 3)
        self.polyfitMenu.Check(self.po4th.GetId(), degree == 4)
        self.polyfitDegree = degree
        self.CalculateCalibrationCoefficients()


    # ------------------------------------------------------------------------------------
    def ToggleHoldPeaks(self,evt):
        self.holdPeaks = (not self.holdPeaks)
        if self.holdPeaks:
            self.toolbar.SetToolNormalBitmap(ID_HPE, wx.Bitmap('icons/icon_holdpeaks_active.png'))
            self.midPanTop.holdPeaksLabel.SetLabel("Holdpeaks: ON")
        else:
            self.toolbar.SetToolNormalBitmap(ID_HPE, wx.Bitmap('icons/icon_holdpeaks.png'))
            self.midPanTop.holdPeaksLabel.SetLabel("Holdpeaks: OFF")

    # ------------------------------------------------------------------------------------
    def ToggleResidualPlot(self,evt):
        self.calibrationResidualPlotFrameIsOpen = not self.calibrationResidualPlotFrameIsOpen
        if self.calibrationResidualPlotFrameIsOpen:
            self.calibrationResidualPlotFrame = CalibrationResidualPlotFrame(self.panel)
            self.toolbar.SetToolNormalBitmap(ID_SRP, wx.Bitmap('icons/icon_showResiduals_active.png'))
        else:
            self.calibrationResidualPlotFrame.Destroy()
            self.toolbar.SetToolNormalBitmap(ID_SRP, wx.Bitmap('icons/icon_showResiduals.png'))

    # ------------------------------------------------------------------------------------
    def ToggleCameraPreviewFrame(self,evt):
        self.cameraPreviewFrameIsOpen = not self.cameraPreviewFrameIsOpen
        if self.cameraPreviewFrameIsOpen:
            self.cameraPreviewFrame = CameraPreviewFrame(self.panel)
            self.toolbar.SetToolNormalBitmap(ID_SCP, wx.Bitmap('icons/icon_preview_active.png'))
        else:
            self.cameraPreviewFrame.Destroy()
            self.toolbar.SetToolNormalBitmap(ID_SCP, wx.Bitmap('icons/icon_preview.png'))

    # ------------------------------------------------------------------------------------
    def ToggleSavitzkyGolayFilter(self,evt,i):
        if i == 0:
            self.useSavitzkyGolayFilter = (not self.useSavitzkyGolayFilter)
        else:
            if self.useSavitzkyGolayFilter:
                self.savgolFilterPolynomial += 1
                if self.savgolFilterPolynomial == 16:
                    self.savgolFilterPolynomial = 0

        if self.useSavitzkyGolayFilter:
            self.toolbar.SetToolNormalBitmap(ID_SGT, wx.Bitmap('icons/icon_SG_filter_on.png'))
            self.midPanTop.savgolFilterLabel.SetLabel("Savgol filter: "+str(self.savgolFilterPolynomial))
        else:
            self.toolbar.SetToolNormalBitmap(ID_SGT, wx.Bitmap('icons/icon_SG_filter_off.png'))
            self.midPanTop.savgolFilterLabel.SetLabel("Savgol filter: OFF")
        self.toolbar.EnableTool(ID_SGP, self.useSavitzkyGolayFilter)
         
    # ------------------------------------------------------------------------------------
    def ToggleNormalisation(self,evt):

        if (self.showNormalised): # we want to turn off normalising
            self.toolbar.SetToolNormalBitmap(ID_NOR, wx.Bitmap('icons/icon_normalise.png'))
            self.normaliseData = [1] * len(self.intensity) #array for normalising data...full of ones
            self.showNormalised = False
            self.toolbar.EnableTool(ID_CAL, True)
            self.toolbar.EnableTool(ID_DAR, True)
        else: # we want to turn normalising ON
            self.toolbar.SetToolNormalBitmap(ID_NOR, wx.Bitmap('icons/icon_normalise_on.png'))
            #self.normaliseData = self.intensity.copy()
            # make the minimum value 1 so we can normalise
            self.normaliseData = np.array([max(1,i) for i in self.intensityFloat])
            if self.applyDarks:
                nd = self.normaliseData - self.darkData
                self.normaliseData = np.array([max(1,i) for i in nd])
#            self.normaliseData = np.array([int(max(1,i)) for i in self.intensityFloat])
            self.showNormalised = True
            self.toolbar.EnableTool(ID_CAL, False)
            self.toolbar.EnableTool(ID_DAR, False)

        self.SetCalibrationStatus()
        self.resetRunningAverageStDev = True

    # ------------------------------------------------------------------------------------
    def ResetRunningAverage(self,evt):
        self.resetRunningAverageData = True
        self.resetRunningAverageStDev = True
        
    # ------------------------------------------------------------------------------------
    '''
    def generateGrid(self)
    # still needs implementation, makes the code perhasp somewhat faster of tens and fifties are produced directly after calibration
    '''
    
    # ------------------------------------------------------------------------------------
    def showCalibrationLampSetUpFrame(self, evt):
        if not self.CalibrationLampSetUpFrameIsOpen:
            self.CalibrationLampFrame = CalibrationLampSetUpFrame(self.panel)
            self.CalibrationLampSetUpFrameIsOpen = True
        else:
            self.CalibrationLampFrame.Close()
            
    # ------------------------------------------------------------------------------------
    def DetectCam(self):

        # see what camera is available
        # first for Raspberry Pi
        if not platformIsWindows:
            self.cameraIsPi = True
            self.cameraIsUSB = False
            self.cameraIsASI = False
            self.canSetGain = True
            self.canSetExposure = False

            self.cameraName = "Pi-cam"
            if self.cameraGain == 0:
                self.cameraGain = 10
            self.cameraMaxGain = 50

            self.picam2 = Picamera2()
            #need to spend more time at: https://datasheets.raspberrypi.com/camera/picamera2-manual.pdf
            #but this will do for now!
            #min and max microseconds per frame gives framerate.
            #30fps (33333, 33333)
            #25fps (40000, 40000)
            
            self.frameWidth = 2028
            self.frameHeight = 1140
            video_config = picam2.create_video_configuration(main={"format": 'RGB888', "size": (self.frameWidth, self.frameHeight)}, controls={"FrameDurationLimits": (33333, 33333)})
            self.picam2.configure(video_config)
            self.picam2.start()
            
            #Change analog gain
            #picam2.set_controls({"AnalogueGain": 10.0}) #Default 1
            #picam2.set_controls({"Brightness": 0.2}) #Default 0 range -1.0 to +1.0
            #picam2.set_controls({"Contrast": 1.8}) #Default 1 range 0.0-32.0

        # then for Windows
        else:
            self.cameraIsPi = False
            # priority is given to ZWO ASI cameras
            asi.init('./ASI SDK/lib/x64/ASICamera2.dll')
            print("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^: you may ignore this warning, PySpectrometer3 is not affected!")
            # how many ZWO cameras are found?
            self.num_cameras = asi.get_num_cameras()
            
            # if no ZWO ASI cameras are found try to find an USB one
            if self.num_cameras == 0:
                self.cameraName = "USB-cam"
                self.cameraIsUSB = True
                self.cameraIsASI = False
                self.cameraGain = 0
                self.cameraMaxGain = 0
                self.canSetGain = False
                self.canSetExposure = False
                if VERBOSE:
                    print('No ASI cameras found, using USB instead')
                    
            # but if ZWO ASI is present use that
            else:
                if VERBOSE:
                    print("Number of ASI cameras found: "+str(asi.get_num_cameras()))
                self.cameraIsUSB = False
                self.cameraIsASI = True
                self.canSetGain = True
                self.canSetExposure = True
                self.ASI_camera_id = 0  # use first camera from list
                self.ASI_camera = asi.Camera(self.ASI_camera_id)
                self.ASI_camera_info = self.ASI_camera.get_camera_property()
                self.ASI_controls = self.ASI_camera.get_controls()
                self.ASI_Gain = self.ASI_camera.get_controls()['AutoExpMaxGain']['DefaultValue']
                self.ASI_Gain_Max = self.ASI_camera.get_controls()['AutoExpMaxGain']['MaxValue']
#                ASI_Exposure = ASI_camera.get_controls()['AutoExpMaxExpMS']['DefaultValue']
#                ASI_Exposure_Max = ASI_camera.get_controls()['AutoExpMaxExpMS']['MaxValue']
                self.ASI_Exposure_Auto = True
                self.ASI_Frame_MaxWidth = self.ASI_camera_info['MaxWidth']
                self.ASI_Frame_MaxHeight = self.ASI_camera_info['MaxHeight']
                if self.cameraGain == 0:
                    self.cameraGain = self.ASI_camera.get_controls()['Gain']['DefaultValue']
                self.cameraMaxGain = self.ASI_camera.get_controls()['Gain']['MaxValue']
                if self.cameraExposure == 0:
                    self.cameraExposure = self.ASI_camera.get_controls()['AutoExpMaxExpMS']['DefaultValue']
                self.cameraMaxExposure = self.ASI_camera.get_controls()['AutoExpMaxExpMS']['MaxValue']
                if VERBOSE:
                    print("ASI camera found: "+self.ASI_camera_info['Name'])
                    print("Camera Exposure: "+str(self.cameraExposure)+" "+str(self.cameraMaxExposure))
                self.cameraName = self.ASI_camera_info['Name']

        self.SetTitle('PySpectrometer3 - '+self.cameraName)
        self.midPanTop.cameraGainLabel.SetLabel("Gain: {}/{}".format(self.cameraGain,self.cameraMaxGain))
        self.midPanTop.cameraExposureLabel.SetLabel("Exposure: {}/{}ms".format(self.cameraExposure,self.cameraMaxExposure))


    # ------------------------------------------------------------------------------------
    def StartCam(self):

        self.statusbar.SetStatusText("Starting camera...")
        if self.cameraIsPi:
            self.cameraIsMono = False
            self.cameraBitDepth = 8
            ret = True
            self.capframe = self.picam2.capture_array()
            self.midPanTop.cameraBitDepthLabel.SetLabel("Bit depth: 8")

        if self.cameraIsUSB:
            self.cameraIsMono = False
            self.cameraBitDepth = 8
            self.cap = cv2.VideoCapture(0)
            ret, self.capframe = self.cap.read()
            self.midPanTop.cameraBitDepthLabel.SetLabel("Bit depth: 8")

        if self.cameraIsASI:
            # Use minimum USB bandwidth permitted
            self.ASI_camera.set_control_value(asi.ASI_BANDWIDTHOVERLOAD, self.ASI_camera.get_controls()['BandWidth']['MinValue'])
            
            # Set some sensible defaults. They will need adjusting depending upon
            # the sensitivity, lens and lighting conditions used.
            self.ASI_camera.disable_dark_subtract()
            
            self.ASI_camera.set_control_value(asi.ASI_GAIN, self.cameraGain)
            #self.ASI_camera.set_control_value(asi.ASI_EXPOSURE, self.cameraExposure*1000,auto=self.ASI_Exposure_Auto) # microseconds
            self.ASI_camera.set_control_value(asi.ASI_EXPOSURE, self.cameraExposure*1000)
            self.ASI_camera.set_control_value(asi.ASI_WB_B, 99)
            self.ASI_camera.set_control_value(asi.ASI_WB_R, 75)
            self.ASI_camera.set_control_value(asi.ASI_GAMMA, 50)
            self.ASI_camera.set_control_value(asi.ASI_BRIGHTNESS, 50)
            self.ASI_camera.set_control_value(asi.ASI_FLIP, 0)
            
            #ASI_camera.set_control_value(asi.ASI_EXPOSURE,controls['Exposure']['DefaultValue'],auto=True)
            #ASI_camera.set_control_value(asi.ASI_EXPOSURE,100000,auto=True)
            
# IN CASE WE WANT TO IMPLEMENT CROPPING
#            Xorg = int(ASI_Frame_MaxWidth/2-frameWidth/2)
#            Yorg = int(ASI_Frame_MaxHeight/2-frameHeight/2)
            
            '''
            ASI_IMG_RAW8 // Each pixel is an 8 bit (1 byte) gray level
            ASI_IMG_RGB24 // Each pixel consists of RGB, 3 bytes totally (color cameras only)
            ASI_IMG_RAW16 // 2 bytes for every pixel with 65536 gray levels
            ASI_IMG_Y8 // mono chrome mode 1 byte every pixel (color cameras only)
            
            The RAW16 produces mono images only and RGB24 is 8 bits per Red, Green, and Blue.
            Everything in ZWO's documentation says RAW16 images sizes are width * height * 2 and RGB24 is width * height * 3.
            '''
            
            self.cameraBitDepth = self.ASI_camera_info['BitDepth']
#            print("Camera Bit-Depth: ",self.cameraBitDepth)
            if self.ASI_camera_info['IsColorCam']:
                self.cameraIsMono = False
                self.ASI_camera.set_image_type(asi.ASI_IMG_RGB24)
                self.cameraBitDepth = 8 # overruling the camera response to avoid 12bit graph
# IN CASE WE WANT TO IMPLEMENT CROPPING
#                ASI_camera.set_roi(Xorg,Yorg,frameWidth,frameHeight,1,asi.ASI_IMG_RGB24)
                self.ASI_camera.set_control_value(asi.ASI_WB_B,self.ASI_controls['WB_B']['DefaultValue'],auto=True)
                self.ASI_camera.set_control_value(asi.ASI_WB_R,self.ASI_controls['WB_R']['DefaultValue'],auto=True)
            else:
                self.cameraIsMono = True
                if self.cameraBitDepth == 8:
                    self.ASI_camera.set_image_type(asi.ASI_IMG_RAW8)
                elif self.cameraBitDepth >= 12:
                    self.ASI_camera.set_image_type(asi.ASI_IMG_RAW16)
            self.midPanTop.cameraBitDepthLabel.SetLabel("Bit depth: "+str(self.cameraBitDepth))
            self.ASI_camera.start_video_capture()
            ret = True
            self.capframe = self.ASI_camera.capture_video_frame()

        self.peakThreshold = int((2**self.cameraBitDepth)*0.01) # 1% of bit depth
        
        self.cameraHeight, self.cameraWidth = self.capframe.shape[:2]
        if VERBOSE:
            print ("Dark comparison: Camera width = {}px, dark width = {}px".format(self.cameraWidth, len(self.darkData)))
        if len(self.darkData)!=self.cameraWidth and self.applyDarks:
            dial = wx.MessageDialog ( self, 'Dark not matching camera width and will thus be removed', 'Notification', wx.OK)
            dial.ShowModal()
            self.applyDarks = False
        self.midPanTop.cameraCaptureWidthLabel.SetLabel("Width: "+str(self.cameraWidth)+"px")
        self.intensity = [0] * self.cameraWidth
        self.intensityFloat = [0] * self.cameraWidth
        
        # Get the size of the drawing area in pixels.
        self.windowWidth, self.windowHeight = self.midPanBottom.GetSize()
        # Create BufferBmp and set the same size as the drawing area.
        self.BufferBmp = wx.Bitmap(self.windowWidth, self.windowHeight)

        #blank image for Waterfall
        self.waterfall = np.zeros([self.windowHeight,self.cameraWidth,3],dtype=np.uint8) 
        self.waterfall.fill(0) #fill black

        self.captureStarted = True
        self.toolbar.EnableTool(ID_SAV, True)
        self.toolbar.EnableTool(ID_DAR, True)
        self.toolbar.EnableTool(ID_NOR, True)
        self.toolbar.EnableTool(ID_CVE, True)
        self.toolbar.EnableTool(ID_CAL, True)
        self.toolbar.EnableTool(ID_FPL, True)
        self.toolbar.EnableTool(ID_FMI, True)
        self.toolbar.EnableTool(ID_CPL, True)
        self.toolbar.EnableTool(ID_CMI, True)
        self.toolbar.EnableTool(ID_MEG, True)
        self.toolbar.EnableTool(ID_MEB, True)
        self.toolbar.EnableTool(ID_SGT, True)
        self.toolbar.EnableTool(ID_SGP, self.useSavitzkyGolayFilter)
        self.toolbar.EnableTool(ID_HPE, True)
        self.toolbar.EnableTool(ID_SRP, True)
        self.toolbar.EnableTool(ID_SCP, True)
        self.toolbar.EnableTool(ID_RES, True)
        self.timer.Start(100)
        self.statusbar.SetStatusText("Ready...")

    # ------------------------------------------------------------------------------------
    def OnTimer(self, event):
        
        if event.GetId() == self.midPanBottom.ID_TIMER:

            # get a frame from the camera
            if self.cameraIsPi:
                self.capframe = picam2.capture_array()
            elif self.cameraIsUSB:
                ret, self.capframe = self.cap.read()
            elif self.cameraIsASI:
                ret = True
                self.capframe = self.ASI_camera.capture_video_frame()
            else:
                print ("No camera present? Exiting...")
                exit()

            # and start processing it
            captureHeight, captureWidth = self.capframe.shape[:2]
            if (captureHeight * captureWidth) > 0: # we have a valid image (hopefully)

                # create a crop around the centre of the image
                if self.cameraIsMono: # we make the image a colour one to keep further processing uniform
                    frame = self.capframe.astype(np.uint16)
                    visframe = cv2.cvtColor(self.capframe,cv2.COLOR_GRAY2RGB) # this works, but is 8bit
                else: # colour images are kept colour
                    frame = cv2.cvtColor(self.capframe, cv2.COLOR_BGR2RGB)
                    visframe = frame.copy()
                frameHeight, frameWidth = frame.shape[:2]
                
                # check if we need to show a full size preview
                if self.cameraPreviewFrameIsOpen:
                # indicate the sample height area in the captured frame
                    vf = visframe.copy()
                    # for visualisation captured 16bit mono data we need to reduce the bit-depth to 8bit (your screen is 8bit anymay)
                    if self.cameraIsMono and self.cameraBitDepth > 8:
                        vf = (vf/256).astype('uint8')
                    cv2.line(vf, (0,int(frameHeight/2-self.sampleHeight/2-self.verticalOffset)), (frameWidth,int(frameHeight/2-self.sampleHeight/2-self.verticalOffset)), (255,255,255), 1,cv2.LINE_AA)
                    cv2.line(vf, (0,int(frameHeight/2+self.sampleHeight/2-self.verticalOffset)), (frameWidth,int(frameHeight/2+self.sampleHeight/2-self.verticalOffset)), (255,255,255), 1,cv2.LINE_AA)
                    self.cameraPreviewFrame.img = wx.Bitmap.FromBuffer(captureWidth, captureHeight, vf).ConvertToImage()
                    self.cameraPreviewFrame.update_bitmap()
                    self.cameraPreviewFrame.on_resize(None)

                if self.verticalOffset == 0  and self.intensityMax > 2**self.cameraBitDepth * 0.7 and self.intensityMax < 2**self.cameraBitDepth * 0.9:
                    self.verticalOffset = self.getVerticalOffset(self.capframe)

                y=int((frameHeight/2)-self.verticalOffset-self.cropHeight/2) #origin of the vertical crop
                croppedFrame = frame[y:y+self.cropHeight, :].copy()
                croppedVisFrame = visframe[y:y+self.cropHeight, :].copy()
        
                # create a DC for graph drawing
                memdc = wx.MemoryDC()
                memdc.SelectObject(self.BufferBmp)

                # actually process the crop
                self.processFrame(memdc,croppedFrame)
                del memdc # need to get rid of the MemoryDC before Update() is called.

                # for visualisation captured 16bit mono data we need to reduce the bit-depth to 8bit (your screen is 8bit anymay)
                if self.cameraIsMono and self.cameraBitDepth > 8:
                    croppedVisFrame = (croppedVisFrame/256).astype('uint8')

                # update and draw the graph
                self.midPanBottom.BufferBmp = self.BufferBmp
                self.midPanBottom.UpdateDrawing(self)

                # show average intensity at the left
                # line-types: https://answers.opencv.org/question/217720/what-does-line_4-line_8-mean/
                sumOfRows2D = np.sum(croppedVisFrame, axis=1)
                sumOfRows = np.sum(sumOfRows2D, axis=1)
                maxVal = np.median(sumOfRows)
                minVal = min(sumOfRows)
                y = 0
                span = 100
                oldX = int((sumOfRows[0]-minVal)/max(maxVal/10,(maxVal-minVal))*span)+10
                oldY = 0
                for sor in sumOfRows:
                    x = int((sor-minVal)/max(maxVal/10,(maxVal-minVal))*span)+10
                    cv2.line(croppedVisFrame, (x, y), (oldX, oldY), wx.YELLOW, 1, cv2.LINE_AA)
                    oldX = x
                    oldY = y
                    y += 1

                # indicate the sample height area in the captured frame
                cv2.line(croppedVisFrame, (0,int(self.cropHeight/2-self.sampleHeight/2)), (frameWidth,int(self.cropHeight/2-self.sampleHeight/2)), (255,255,255), 1,cv2.LINE_4)
                cv2.line(croppedVisFrame, (0,int(self.cropHeight/2+self.sampleHeight/2)), (frameWidth,int(self.cropHeight/2+self.sampleHeight/2)), (255,255,255), 1,cv2.LINE_4)

                cf = croppedVisFrame
                self.midPanCentre.img = wx.Bitmap.FromBuffer(captureWidth, self.cropHeight, cf).ConvertToImage()
#                self.midPanCentre.img = croppedFrame.ConvertToImage()
                self.midPanCentre.update_bitmap()
        else:
            event.Skip()

    # ------------------------------------------------------------------------------------
    def OnExit(self, evt):
        if platformIsWindows:
            if evt.GetEventType() == wx.EVT_CLOSE.typeId:
                dial = wx.MessageDialog ( self, 'Are you sure you want to exit?', 'Question', wx.YES_NO | wx.ICON_EXCLAMATION )
                
                if dial.ShowModal() == wx.ID_NO:
                    return
        if self.FWHM_DataFrameIsOpen:
            self.FWHM_DataFrame.Close()
        self.SavePrefs()
        print("Bye!")
        self.timer.Stop()
        self.Destroy()

    # ------------------------------------------------------------------------------------
    def Gaussian(self, x, amp, cen, wid, cont):
        with warnings.catch_warnings(record=True) as w:
            return amp * np.exp(-(x-cen)**2 / wid) + cont
        if len(w) > 0:
#            print("Oops (Guassian fit)")
            pass
            
    # ------------------------------------------------------------------------------------
    def getVerticalOffset(self, theFrame):
        sumOfRows2D = np.sum(theFrame, axis=1)
        if self.cameraIsMono:
            sumOfRows = sumOfRows2D
        else:
            sumOfRows = np.sum(sumOfRows2D, axis=1)
        nRows = len(sumOfRows)
        criterion = np.amax(sumOfRows)*0.95

        # only calculate the offset when the maximum intesity is more than 70% and less than 90%
        firstRow = -1
        lastRow = -1
        for i in range(0,nRows):
            if firstRow == -1 and sumOfRows[i] >= criterion:
                firstRow = i
            if firstRow != -1 and lastRow == -1 and sumOfRows[i] <= criterion and (i - firstRow) > 20:
                lastRow = i
        offsetFromStart = int(max(nRows/4,(firstRow+lastRow)/2))
        offsetFromCentre = int(len(sumOfRows)/2-offsetFromStart)
#        print("First and last rows: {} {}, offset from start = {}, offset from centre = {}".format(firstRow,lastRow,offsetFromStart,offsetFromCentre))
        print("Spectrum vertically centred, offset now is {}px from centre.".format(offsetFromCentre))
        return offsetFromCentre

    # ------------------------------------------------------------------------------------
    def RecentreSpectrum(self,evt):
        if self.intensityMax > 2**self.cameraBitDepth * 0.9:
            print("For centring the spectrum ensure the max intensity ({:.0f} ADU) is below {:.0f} ADU.".format(self.intensityMax,2**self.cameraBitDepth * 0.9))
        else:
            self.verticalOffset = self.getVerticalOffset(self.capframe)

    # ------------------------------------------------------------------------------------
    def countColours(self,data,string):
#        w,h = data.shape
        colours = [0] * 65536 #(w * h)
        idx = 0
        row = 0
        if string == "image":
            for pxX in data:
                col = 0
                for pxY in pxX:
                    colours[int(pxY)] += 1
                    idx += 1
                    col += 1
                row += 1
        else:
            colours = [0] * (40 * 1936) #(w * h)
            for pxY in data:
                colours[int(pxY)] += 1
                idx += 1
                col += 1
        idx2 = 0
        for c in colours:
            if c > 0:
                idx2 += 1
        print("{}: pixels: {}, colours: {}, rows: {}, cols: {}".format(string,idx, idx2, row, col))
    # ------------------------------------------------------------------------------------
    def processFrame(self, theDc, theFrame):
        frameHeight, frameWidth = theFrame.shape[:2]
        annotationOffset = 60 #int(self.windowHeight/5) # create a blank area below the graph to annotate the peaks
        currentHoldPeaks = self.holdPeaks and True # copy of self.holdPeaks to avoid it being changed while processing


        theDc.SetBrush(wx.Brush(wx.WHITE, wx.SOLID))
        theDc.Clear()

        # make mono-image of colour-image to get the intensities
        if self.cameraBitDepth > 8:
            bwimage = theFrame.astype(np.uint16)
#            self.countColours(bwimage,"image")
#            print(type(bwimage[0][0])) # check bit-depth
        else:
            bwimage = cv2.cvtColor(theFrame,cv2.COLOR_BGR2GRAY)
        
        # read the intensities from the mono-image
        oldIntensity = self.intensity.copy()
        rows,cols = bwimage.shape
        halfway = int(rows/2 - self.sampleHeight/2)
        for i in range(cols):
        
            data = np.mean(bwimage[halfway:halfway+self.sampleHeight, i:i+1])
            if self.cameraBitDepth == 8:
                data = np.uint8(int(data))
            elif self.cameraBitDepth == 12:
                data = np.uint16(int(data))>>4 # bit shift right four positions, is same as " / 65535 * 4095": reduction from 16 bit to 12 bit (16**2-1) * (12**2-1)
            else: # cameraBitDepth == 16!
                data = np.uint16(int(data))
            # we store data in a seond array intensityTmp to preserve decimal average values
            if currentHoldPeaks:
                self.intensityFloat[i] = max(data, self.intensityFloat[i])*1.0
            else:
                if self.resetRunningAverageData:
                    self.intensityFloat[i] = data * 1.0
                else:
                    self.intensityFloat[i] = (self.intensityFloat[i]*(self.runningAvgAmount-1)+data) / self.runningAvgAmount

        if self.resetRunningAverageData:
            self.resetRunningAverageData = False
            self.runningAvgCounter = 0
        if self.runningAvgCounter<self.runningAvgAmount:
            self.runningAvgCounter += 1
            if self.runningAvgCounter < self.runningAvgAmount:
                self.midPanTop.runningAvgLabel.SetForegroundColour((255,0,0))
                self.midPanTop.runningAvgLabel.SetLabel("Running AVG: {}/{}".format(self.runningAvgCounter,self.runningAvgAmount))
            else:
                self.midPanTop.runningAvgLabel.SetForegroundColour((0,0,0))
                self.midPanTop.runningAvgLabel.SetLabel("Running AVG: {}/{} ".format(self.runningAvgCounter,self.runningAvgAmount))


#        self.countColours(self.intensityFloat,"profile")

        # now copy the float running average intensity as int to intensity array and ensure that the minum value is 1 so that the data can be used to normalise
        self.intensity = np.array(self.intensityFloat.copy())

        # now apply darks to the data
        if self.applyDarks:
            self.intensity = self.intensity - self.darkData

        # get the maximum intensity, required for determining the vertical offset
        self.intensityMax = max(self.intensity)

        # now normalise the data
        if self.showNormalised:
            self.intensity = np.divide(self.intensity,self.normaliseData).copy()

        # apply SavitzkyGolayFilter
        if self.useSavitzkyGolayFilter:
            self.intensity = savitzky_golay(self.intensity,17,self.savgolFilterPolynomial)
            self.intensity = np.array(self.intensity)

        # check the differenceStDev with last average as indication of stability
        if self.showNormalised:
            factor = 100
        else:
            factor = 1
#            factor = 100*(2**self.cameraBitDepth)
        differenceStDev = np.std(self.intensity - oldIntensity)*factor
        if self.resetRunningAverageStDev:
            self.runningAverageStDev = differenceStDev * 0
            self.resetRunningAverageStDev = False
        else:
            self.runningAverageStDev = (self.runningAverageStDev * (self.runningAverageStDevAmount-1) + differenceStDev) / self.runningAverageStDevAmount
        self.runningAverageStDevRegressionArrayLastValues[0] = self.runningAverageStDev
        self.runningAverageStDevRegressionArrayLastValues = np.roll(self.runningAverageStDevRegressionArrayLastValues, -1)
        res = stats.linregress(self.runningAverageStDevRegressionArrayNumbers, self.runningAverageStDevRegressionArrayLastValues)
        self.runningAverageStDevRegressionCriterion = 2**self.cameraBitDepth/self.cameraWidth**0.5/self.sampleHeight/self.runningAvgAmount/self.runningAverageStDevAmount
#        print("Slope:Criterion = {0:.5f}:{1:.5f}".format(abs(res.slope),self.runningAverageStDevRegressionCriterion),2**self.cameraBitDepth)
        if (abs(res.slope) < self.runningAverageStDevRegressionCriterion):
            self.midPanTop.stabilityLabel.SetForegroundColour((0,0,0))
            self.midPanTop.stabilityLabel.SetLabel("dStDev: {0:.2f}ADU".format(self.runningAverageStDev)) # we update the label WITHOUT an additional space behind "ADU" to ensure the StaticText is updated.
        else:
            self.midPanTop.stabilityLabel.SetForegroundColour((255,0,0))
            self.midPanTop.stabilityLabel.SetLabel("dStDev: {0:.2f}ADU ".format(self.runningAverageStDev)) # we update the label WITH an additional space behind "ADU" to ensure the StaticText is updated.


        # Display a graticule calibrated with cal data
        # vertial lines every whole 10nm and every 50nm a bit darker
        idx = 0
        oldWl = self.wavelengthData[idx]
        oldX = 0
        dx, dy = (0,0)
        canDoNext = True
        for i in range(0,len(self.intensity)-2):
            sign = np.sign(self.wavelengthData[idx]-self.wavelengthData[idx+1])
            wl = round(self.wavelengthData[idx]+sign*0.45,0) # by adding/subtracting 0.45nm to the wavelengthdata the lines get rounded correctly and drawn in the right place
            if wl%10 == 0 and canDoNext:
                x = self.getGraphX( self.windowWidth, frameWidth, idx)
                if wl%50 == 0:
                    theDc.SetPen(wx.Pen(wx.Colour(100,100,100), 1, wx.SOLID))
                    theDc.DrawLine( x, 0, x, self.windowHeight-annotationOffset)
                    theStr = "{0:.0f}nm".format(self.wavelengthData[idx])
                    dx, dy = theDc.GetTextExtent(theStr)
                    theDc.SetPen(wx.Pen(wx.WHITE, 1, wx.SOLID))
                    theDc.DrawRectangle(x-int(dx/2),1,dx,dy)
                    theDc.DrawText( theStr, (x-int(dx/2), 1))
                    oldX = x
                else:
                    theDc.SetPen(wx.Pen(wx.Colour(200,200,200), 1, wx.SOLID))
                    if (x-oldX) < (dx/2):
                        dY = dy+1
                    else:
                        dY = 0
                    theDc.DrawLine( x, dY, x, self.windowHeight-annotationOffset)
                oldWl = wl
                canDoNext = False
            idx += 1
            if abs(oldWl-wl) > 0.1:
                canDoNext = True
        
        # horizontal lines
        theDc.SetPen(wx.Pen(wx.Colour(200,200,200), 1, wx.SOLID))
        theDc.SetTextForeground(wx.Colour(100,100,100))
        if self.showNormalised:
            theRange = 200
            theInterval = 20
            theDivider = 100
            theFormat = "{0:.0f}%"
        else:
            theRange = 2**self.cameraBitDepth+1
            theInterval = int(theRange/8)
            theDivider = 1
            theFormat = "{0:.0f}ADU"
        for i in range (0, theRange, theInterval):
            theStr = theFormat.format(i)
            dx, dy = theDc.GetTextExtent(theStr)
            y = self.getGraphY( self.windowHeight, annotationOffset, i/theDivider)
            theDc.SetPen(wx.Pen(wx.WHITE, 1, wx.SOLID))
            theDc.DrawRectangle(2,y,dx,-dy)
            theDc.DrawRectangle(self.windowWidth-2,y,-dx,-dy)
            theDc.SetPen(wx.Pen(wx.Colour(200,200,200), 1, wx.SOLID))
            theDc.DrawLine( 0, y, self.windowWidth, y)
            theDc.DrawText( theStr, (2, y-dy-1))
            theDc.DrawText( theStr, (self.windowWidth-dx-2, y-dy-1))
        if self.showNormalised:
            y = self.getGraphY( self.windowHeight, annotationOffset, 100/theDivider)
            theDc.SetPen(wx.Pen(wx.Colour(100,100,100), 1, wx.SOLID))
            theDc.DrawLine( 0, y, self.windowWidth, y)
        
        # draw an intensity line of 80% ADU (above we may expect the camera to be non-linear)
        y = self.getGraphY( self.windowHeight, annotationOffset, (2**self.cameraBitDepth+1)*0.8)
        theStr = "80%"
        dx, dy = theDc.GetTextExtent(theStr)
        theDc.SetPen(wx.Pen(wx.WHITE, 1, wx.SOLID))
        theDc.DrawRectangle(2,y,dx,-dy)
        theDc.DrawRectangle(self.windowWidth-2,y,-dx,-dy)
        pen = wx.Pen(wx.RED, 1,  wx.USER_DASH)
        pen.SetDashes([7, 7])
        theDc.SetPen(pen)
        theDc.SetTextForeground(wx.RED)
        theDc.DrawLine(0, y, self.windowWidth, y)
        theDc.DrawText( theStr, (2, y-dy-1))
        theDc.DrawText( theStr, (self.windowWidth-dx-2, y-dy-1))
 
        '''
        # this waterfall display still need a lot of work
        # most likely need to keep all data in memory as here we start with a freh canvas each time
        # the Blit method may work to copy part of the memDc
        dispWaterfall = False
        # create waterfall data....
        if dispWaterfall:
            #create an empty array for the data
            wdata = np.zeros([1,self.cameraWidth,3],dtype=np.uint16)
            index=0
            for i in self.intensity:
                rgb = wavelength_to_rgb(round(self.wavelengthData[index]))#derive the color from the wavelenthData array
                luminosity = self.intensity[index]/255
                b = max(0,int(round(rgb[0]*luminosity)))
                g = max(0,int(round(rgb[1]*luminosity)))
                r = max(0,int(round(rgb[2]*luminosity)))
                #print(b,g,r)
                #wdata[0,index]=(r,g,b) #fix me!!! how do we deal with this data??
                wdata[0,index]=(r,g,b)
                index+=1
            #bright and contrast of final image
            contrast = 2.5
            brightness =10
            wdata = cv2.addWeighted( wdata, contrast, wdata, 0, brightness)
            self.waterfall = np.insert(self.waterfall, 0, wdata, axis=0) #insert line to beginning of array
            self.waterfall = self.waterfall[:-1].copy() #remove last element from array
            theDc.Blit(0, 1, self.cameraWidth, self.windowHeight-1, theDc, 0, 0)#, logicalFunc=COPY, useMask=False, xsrcMask=DefaultCoord, ysrcMask=DefaultCoord)
        '''

       # now draw the intensity data....
        index=0
        x0 = -1
        y0 = -1
        penThickness = math.floor(self.windowWidth/frameWidth)+1
        for i in self.intensity:
            rgb = wavelength_to_rgb(round(self.wavelengthData[index]))#derive the color from the wvalenthData array
            r = rgb[0]
            g = rgb[1]
            b = rgb[2]
            #origin is top left.
            x = self.getGraphX( self.windowWidth, frameWidth, index)
            y = self.getGraphY( self.windowHeight, annotationOffset, i)
            theDc.SetPen(wx.Pen(wx.Colour(r,g,b), penThickness, wx.SOLID))
            theDc.DrawLine(x, self.windowHeight-annotationOffset, x, y)
            index+=1
        index=0
        theDc.SetPen(wx.Pen(wx.BLACK, 1, wx.SOLID))
        for i in self.intensity:
            x = self.getGraphX( self.windowWidth, frameWidth, index)
            y = self.getGraphY( self.windowHeight, annotationOffset, i)
            if index > 0:
                theDc.DrawLine(x0, y0, x, y)
            x0 = x
            y0 = y
            index+=1

        # draw the wavelength calibration graph
        '''
        theDc.SetPen(wx.Pen(wx.Colour(100,100,100), 1, wx.SOLID))
        x0 = 0
        y0 = 0
        index = 0
        for wl in self.wavelengthData:
            x = self.getGraphX( self.windowWidth, frameWidth, index)
            y = self.getGraphY( self.windowHeight, annotationOffset, wl)
            if index > 0:
                theDc.DrawLine(x0, y0, x, y)
            x0 = x
            y0 = y
            index+=1
        '''

        # show the calculation-zone and FWHM of a selected peak
        pen = wx.Pen(wx.Colour(220,220,220), 1,  wx.USER_DASH)
        pen.SetDashes([5, 10])
        theDc.SetPen(pen)
        if self.doFWHM_Measurement:
            # draw first region line
            x = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_RangeStart)
            theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
            
        if self.FWHM_MeasurementRightDone:
            # calculate FWHM
            suc = False
            try:
                if self.doGaussianFWHM_Measurement:
                    x_crop, bestFit, cov = self.CalculateGaussianFWHM()
                else:
                    self.CalculateBlockFWHM()
                suc = True
            except:
#                print("Oops")
                pass
            
            if suc==True:
                # draw second region line
                x = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_RangeEnd)
                theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
                # calculate three lines indicating FWHM
                # first the diagonal representing the average intensity of surrounding data
                x1 = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_RangeStart)
                x2 = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_RangeEnd)
                y1 = self.getGraphY(  self.windowHeight, annotationOffset, self.intensity[self.FWHM_RangeStart])
                y2 = self.getGraphY(  self.windowHeight, annotationOffset, self.intensity[self.FWHM_RangeEnd])
                theDc.DrawLine(x1, y1, x2, y2)

                # then the two FWHM lines
                if self.doGaussianFWHM_Measurement:
                    pen = wx.Pen(CURSOR_COLOUR_FWHM_GAUSSIAN, 2,  wx.USER_DASH)
                else:
                    pen = wx.Pen(CURSOR_COLOUR_FWHM_BLOCK, 2,  wx.USER_DASH)
                pen.SetDashes([7, 7])
                x = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_PeakPosLeft)
                theDc.SetPen(wx.Pen(wx.WHITE, 2, wx.SOLID))
                theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
                theDc.SetPen(pen)
                theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
                x = self.getGraphX( self.windowWidth, frameWidth, self.FWHM_PeakPosRight)
                theDc.SetPen(wx.Pen(wx.WHITE, 2, wx.SOLID))
                theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
                theDc.SetPen(pen)
                theDc.DrawLine(x, self.windowHeight-annotationOffset, x, 0)
    
                if self.doGaussianFWHM_Measurement:
                    index = 0
                    x0 = 0
                    y0 = 0
                    test_x = np.linspace(self.FWHM_RangeStart,self.FWHM_RangeEnd,self.FWHM_RangeEnd-self.FWHM_RangeStart+1)
                    fittedCurve = self.Gaussian(test_x, *bestFit)
                    theDc.SetPen(wx.Pen(wx.RED, 2, wx.SOLID))
                    for x in x_crop:
                        x = self.getGraphX( self.windowWidth, frameWidth, x)
                        y = self.getGraphY(  self.windowHeight, annotationOffset, fittedCurve[index])
                        if index>0:
                            theDc.DrawLine(x0, y0, x, y)
                        index += 1
                        x0 = x
                        y0 = y
                
        # draw calibration data
        if time.time() - self.calibrationStartTime < self.maxCalibrationShowTime or self.CalibrationLampSetUpFrameIsOpen:
            idx = 0
            theDc.SetPen(wx.Pen(wx.Colour(255,150,0), 1, wx.SOLID))
            theDc.SetTextForeground(wx.Colour(255,150,0))
            textoffset = 2
            if self.isCalibrated:
                for ocp in self.originalCalibrationPixels:
                    x = self.getGraphX( self.windowWidth, frameWidth, ocp)
                    y = self.windowHeight - annotationOffset
                    theStr = str(self.correspondingWavelengths[idx])+"nm"
                    dx, dy = theDc.GetTextExtent(theStr)
        #            print(idx, cwl, x, y)
                    theDc.SetPen(wx.Pen(wx.Colour(255,150,0), 1, wx.SOLID))
                    theDc.DrawLine(x, 0, x, y)
                    theDc.SetPen(wx.Pen(wx.WHITE, 1, wx.SOLID))
                    theDc.DrawRectangle(x+textoffset, 15, dy-4, dx+5)
                    theDc.DrawRotatedText(theStr,(x+dy+textoffset,15),270)
                    idx += 1
            pen = wx.Pen(wx.Colour(255,150,0), 1,  wx.USER_DASH)
            pen.SetDashes([7, 7])
            theDc.SetPen(pen)
#            theDc.SetPen(wx.Pen(wx.Colour(255,150,0), 1, wx.SOLID))
            y = self.getGraphY(  self.windowHeight, annotationOffset, max(self.intensity)*self.calibrationDetectionLevel)
            theDc.DrawLine(0, y, self.windowWidth, y)
        
        
        # draw peak data
        textoffset = 18 #int(frameWidth/200) #20 #12
        # define peak threshold  
        theDc.SetPen(wx.Pen(wx.BLACK, 1, wx.SOLID))
        theDc.SetTextForeground(wx.BLACK)
        if self.showNormalised:
            pt = 0.95
        else:
            pt = int(self.peakThreshold) #make sure the data is int.
        # draw the peaks
        indexes = peakIndexes(self.intensity, thres=pt/max(self.intensity), min_dist=self.peakMinDist)
        for i in indexes:
            x = self.getGraphX( self.windowWidth, frameWidth, i)
            y = self.getGraphY( self.windowHeight, annotationOffset, self.intensity[i])
            wavelength = round(self.wavelengthData[i],1)
            # annotate the text 90 degrees rotated
            if self.isCalibrated:
                theDc.DrawRotatedText(str(wavelength)+'nm',(x+textoffset,self.windowHeight-annotationOffset+5),270)
            else:
                theDc.DrawRotatedText(str(i)+'px',(x+textoffset,self.windowHeight-annotationOffset+5),270)
            # draw the lines
            theDc.DrawLine(x, self.windowHeight, x, y+10)
        
        # draw the cursor
        if self.dataIdx>=0:
            if self.doFWHM_Measurement:
                if self.doGaussianFWHM_Measurement:
                    theDc.SetPen(wx.Pen(CURSOR_COLOUR_FWHM_GAUSSIAN, 1, wx.SOLID))
                    theDc.SetTextForeground(CURSOR_COLOUR_FWHM_GAUSSIAN)
                else:
                    theDc.SetPen(wx.Pen(CURSOR_COLOUR_FWHM_BLOCK, 1, wx.SOLID))
                    theDc.SetTextForeground(CURSOR_COLOUR_FWHM_BLOCK)
            else:
                theDc.SetPen(wx.Pen(CURSOR_COLOUR_NORMAL, 1, wx.SOLID))
                theDc.SetTextForeground(CURSOR_COLOUR_NORMAL)
            x = self.getGraphX( self.windowWidth, frameWidth, self.dataIdx)
            theDc.DrawLine(x, self.windowHeight, x, 0)
            wavelength = round(self.wavelengthData[self.dataIdx],1)
            theDc.DrawRotatedText(str(wavelength)+'nm',(x+textoffset,self.windowHeight-annotationOffset+5),270)

    # ------------------------------------------------------------------------------------
    def getGraphX(self, windowWidth, dataWidth, pos):
        x = int(pos/dataWidth*windowWidth)
        return x

    # ------------------------------------------------------------------------------------
    def getGraphY(self, graphHeight, annotationOffset, intensity):
        if (self.showNormalised):
            y = int(graphHeight-annotationOffset-intensity*(graphHeight-annotationOffset)*0.5)
        else:
            y = int(graphHeight-annotationOffset-intensity/(2**self.cameraBitDepth-1)*(graphHeight-annotationOffset)*self.graphFillHeight)
        return y

    # ------------------------------------------------------------------------------------
    '''
    def create_wx_bitmap(self,cv2_image):
        height, width = cv2_image.shape[:2]
        info = np.iinfo(cv2_image.dtype) # Get the information of the incoming image type
        data = cv2_image.astype(np.float64) / info.max # normalize the data to 0 - 1
        data = 255 * data # Now scale by 255
        cv2_image = data.astype(np.uint8)
        cv2_image_rgb = cv2.cvtColor(cv2_image, cv2.COLOR_BGR2RGB)
        return wx.Bitmap.FromBuffer(width, height, cv2_image_rgb)
    '''    


    # ------------------------------------------------------------------------------------
    def CalculateGaussianFWHM(self):
        # here we calculate the FWHM of a selected Gaussian shape, toggling the FWHM function on/off is done in self.ToggleMeasureFWHM(), registering the mouse-click locations self.GetDataIdx()
        i = (self.FWHM_RangeStart + self.FWHM_RangeEnd)/2
        x_crop = range(self.FWHM_RangeStart,self.FWHM_RangeEnd)

        y_crop_object = self.intensity[self.FWHM_RangeStart:self.FWHM_RangeEnd]
        try:
            best_vals_object, covar = curve_fit(self.Gaussian, x_crop, y_crop_object, p0=[100000, i, 10, np.min(y_crop_object)])
        except:
            return None, None, None
        '''
        best_vals_object[0]: peak value relative to base value
        best_vals_object[1]: location of peak value (in pixels here)
        best_vals_object[2]: width of the Gaussian curve, FWHM = 2 x sqrt(2 x ln(2)) * best_vals_object[2] (in pixels here)
        best_vals_object[3]: base value
        
        covar contains the variances of best_vals_object at its main diagonal
        '''
#        print (best_vals_object,covar)
        intTestPxRange = int(min(1,best_vals_object[1]))*2 # the number of pixels over which the dispersion in nm/px will be calculated
        intCentrePxValue = int(best_vals_object[1])
        testRangeWavelengthDiff = self.wavelengthData[intCentrePxValue + intTestPxRange] - self.wavelengthData[intCentrePxValue - intTestPxRange]
        dispersion = testRangeWavelengthDiff / (2 * intTestPxRange)
#        print("FHWM dispersion",dispersion)
        c = math.sqrt(abs(best_vals_object[2]/2))
        FWHM = 2*math.sqrt(2*np.log(2)) * c * dispersion # the 2 in log(2) means we calculate width at max/2, so at half max, see: https://en.m.wikipedia.org/wiki/Gaussian_function
        peak0 = self.wavelengthData[intCentrePxValue]
        peak1 = self.wavelengthData[intCentrePxValue+1]
        peakPos = peak0 + (peak1 - peak0) * (best_vals_object[1] % 1)
#        print("Peak pos vs raw: ",peakPos,self.wavelengthData[intCentrePxValue])
        peakVal = best_vals_object[0]+best_vals_object[3]
        if best_vals_object[0]>best_vals_object[3]:
            str = "Transmission"
        else:
            str = "Absorption"
        self.FWHM_PeakPosLeft = best_vals_object[1] - 2*math.sqrt(2*np.log(2)) * c/2
        self.FWHM_PeakPosRight = best_vals_object[1] + 2*math.sqrt(2*np.log(2)) * c/2
        
        if self.FWHM_DataFrameIsOpen:
            self.FWHM_DataFrame.FWHM_Label1txt.SetLabel(str)
            self.FWHM_DataFrame.FWHM_Label2txt.SetLabel("{0:.2f}nm".format(FWHM))
            self.FWHM_DataFrame.FWHM_Label3txt.SetLabel("{0:.2f}nm".format(peakPos))
            self.FWHM_DataFrame.FWHM_Label4txt.SetLabel("{0:.2f}ADU".format(peakVal))
            self.FWHM_DataFrame.FWHM_Label5txt.SetLabel("{0:.1f}".format(peakPos/FWHM))
        
        return x_crop, best_vals_object, covar

        
    # ------------------------------------------------------------------------------------
    def CalculateBlockFWHM(self):
        # here we calculate the FWHM of a selected block-shape, toggling the FWHM function on/off is done in self.ToggleMeasureFWHM(), registering the mouse-click locations self.GetDataIdx()
        # copy the data fro processing
        selection = self.intensity[self.FWHM_RangeStart:self.FWHM_RangeEnd].copy()
        selectionLength = len(selection)
        if selectionLength <= 1:
            self.FWHM_MeasurementLeftDone = False
            self.FWHM_MeasurementRightDone = False
            return

        # determine min/max and whether the peak is absorption or transmission
        minVal = min(selection)
        maxVal = max(selection)
        minValDiff = abs((self.intensity[self.FWHM_RangeStart]+self.intensity[self.FWHM_RangeEnd])/2-minVal)
        maxValDiff = abs((self.intensity[self.FWHM_RangeStart]+self.intensity[self.FWHM_RangeEnd])/2-maxVal)
        if minValDiff < maxValDiff:
            thePeakIsTransmission = True
            str = "Transmission"
        else:
            thePeakIsTransmission = False
            str = "Absorption"
        
        foundFirst = False
        # calculate Half Maximum
        HM = (minVal+maxVal)/2
        idx = 0
        ppl = 0
        ppr = 0
        peakTotal = 0
        peakTotalIdx = 0
        for s in selection:
            if thePeakIsTransmission:
                # treat the data as a transmission peak
                if (s >= HM):
                    peakTotal += s
                    peakTotalIdx += 1
                if (s >= HM) and (not foundFirst):
                    ppl = self.InterpolateBlockFWHM_Slope(HM,s,idx)
                    foundFirst = True
                if (s <= HM) and (foundFirst):
                    ppr = self.InterpolateBlockFWHM_Slope(HM,s,idx)
                    break # no need to look any further
            else:
                # treat the data as an absorption peak
                if (s <= HM):
                    peakTotal += s
                    peakTotalIdx += 1
                if (s <= HM) and (not foundFirst):
                    ppl = self.InterpolateBlockFWHM_Slope(HM,s,idx)
                    foundFirst = True
                if (s >= HM) and (foundFirst):
                    ppr = self.InterpolateBlockFWHM_Slope(HM,s,idx)
                    break # no need to look any further
            idx += 1
        
        lambda0 = self.wavelengthData[int(ppl)]
        lambda1 = self.wavelengthData[int(ppl)+1]
        lambdaLeft = lambda0 + (lambda1 - lambda0) * (ppl % 1)

        lambda0 = self.wavelengthData[int(ppr)]
        lambda1 = self.wavelengthData[int(ppr)+1]
        lambdaRight = lambda0 + (lambda1 - lambda0) * (ppr % 1)

        # calculate position, intensity and FWHM of peak
        peakPos = (lambdaLeft + lambdaRight) / 2
        peakVal = peakTotal/peakTotalIdx
#        peakVal = self.intensity[int((ppl + ppr)/2)]
        FWHM = lambdaRight - lambdaLeft

        if self.FWHM_DataFrameIsOpen:
            self.FWHM_DataFrame.FWHM_Label1txt.SetLabel(str)
            self.FWHM_DataFrame.FWHM_Label2txt.SetLabel("{0:.2f}nm".format(FWHM))
            self.FWHM_DataFrame.FWHM_Label3txt.SetLabel("{0:.2f}nm".format(peakPos))
            self.FWHM_DataFrame.FWHM_Label4txt.SetLabel("{0:.2f}ADU".format(peakVal))

        self.FWHM_PeakPosLeft = int(ppl)
        self.FWHM_PeakPosRight = int(ppr)

    # ------------------------------------------------------------------------------------
    def InterpolateBlockFWHM_Slope(self,HM,s,idx):
        ds = HM-s
        di = self.intensity[self.FWHM_RangeStart + idx + 1] - self.intensity[self.FWHM_RangeStart + idx]
        if di == 0:
            output = self.FWHM_RangeStart + idx
        else:
            output = self.FWHM_RangeStart + idx + ds/di
        return output

    # ------------------------------------------------------------------------------------
    def Save(self,evt):
        print("Save button pressed")
        Path("./data/").mkdir(parents=True, exist_ok=True)
        now = time.strftime("%Y%m%d--%H%M%S")
        timenow = time.strftime("%H:%M:%S")

        if self.dispWaterfall:
            print("writing waterfall")
            cv2.imwrite("waterfall-" + now + ".png",waterfall_vertical)
        else:
            # ------- saving graph
            print("writing graph")
            self.BufferBmp.SaveFile("./data/graph-" + now + ".png", wx.BITMAP_TYPE_PNG)
    
            # ------- saving raw spectrum image
            print("writing spectrum")
            if self.cameraBitDepth > 8:
                img = cv2.cvtColor(self.capframe, cv2.CV_16U)
            else:
                img = self.capframe
            cv2.imwrite("./data/spectrum-" + now + ".tiff",img)
    
            # ------- saving residual plot
            if self.calibrationResidualPlotFrameIsOpen:
                print("writing residual plot")
                self.calibrationResidualPlotFrame.theCanvas.SaveFile("./data/residualplot-" + now + ".png")
    
            # ------- saving csv file
            print("writing csv")
            f = open("./data/Spectrum-"+now+'.csv','w')
            f.write('#Wavelength,#Intensity\n')
            for x in zip(self.wavelengthData,self.intensity):
                f.write(str(x[0])+','+str(x[1])+'\n')
            f.close()
            
            # ------- saving 1D FITS-profile
            self.SaveFITS(now)
    
        print("All saved!")
        message = "Last Save: "+timenow
        self.midPanTop.saveDataLabel.SetLabel(message)
        
    # ------------------------------------------------------------------------------------
    def SaveFITS(self,ts):
        print("writing FITS")
        
        # set destination
        fitsImageFilenameRaw = "./data/Spectrum-"+ts+"-1D_raw.fits"
        fitsImageFilenameCalibrated = "./data/Spectrum-"+ts+"-1D_calibrated.fits"
        
        # Set export-mode
        exportAsAngstrom = True 
        
        # first we need to create arrays based on equidistant wavelengths as otherwise it cannot be saved
        # this is still done in nanometres
        wvlStart = min(self.wavelengthData)
        wvlEnd = max(self.wavelengthData)
        wvlDispersion = (wvlEnd-wvlStart)/(len(self.wavelengthData)-1)
        wvlEquidist = [] # new wavelength values based on average dispersion (i.e. the list contains a linear table of wavelengths based on the average dispersion)
        intEquidist = [] # array for interpolated intensities
        pixels = [] # array with pixel numbers for raw data output
        wvlCurrent = wvlStart
        idx = 0
        while wvlCurrent<=wvlEnd:
            while self.wavelengthData[idx] < wvlCurrent and idx < (len(self.wavelengthData)-1):
                idx += 1
            interpolatedValue = (wvlCurrent - self.wavelengthData[idx-1])/(self.wavelengthData[idx]-self.wavelengthData[idx-1]) * (self.intensity[idx]-self.intensity[idx-1])+self.intensity[idx-1]
            wvlEquidist.append(wvlCurrent*10) # output must be in Angstrom!
            intEquidist.append(interpolatedValue)
            wvlCurrent += wvlDispersion

        idx = 0
        maxInt = 0
        for i in self.wavelengthData:
            pixels.append(idx)
            if i > maxInt:
                maxInt = i
            idx += 1
        print ("Raw data contains ",idx,"pixels, max value = ",maxInt)
            
        # Write spectrum providing wavelength array, this sets header keywords SIMPLE, BITPIX, NAXIS, NAXIS1, EXTEND, CTYPE1, CRPIX1, CRVAL1 and CDELT1
        # raw data
        pyastro.write1dFitsSpec(fitsImageFilenameRaw, np.array(self.intensityFloat), wvl=np.array(pixels), clobber=True)
        # calibrated data
        pyastro.write1dFitsSpec(fitsImageFilenameCalibrated, np.array(intEquidist), wvl=np.array(wvlEquidist), clobber=True)
        
        # Now re-open the file to set rest of FITS header
        hdul = fits.open(fitsImageFilenameCalibrated, mode='update')
        hdr = hdul[0].header

        # first the standard keywords (https://heasarc.gsfc.nasa.gov/docs/fcg/standard_dict.html)
        hdr.append(('DATE'     , time.strftime("%Y-%m-%dT%H:%M:%S")               , "Date of integration"), end=True)
        hdr.append(('DATE-OBS' , time.strftime("%Y-%m-%dT%H:%M:%S")               , "Date of integration"), end=True)
        hdr.append(('INSTRUME' , self.cameraName)                                 , end=True)
        
        # then non-standard keywords
        # camera type
        if self.cameraIsMono:
            value = "Mono"
        else:
            value = "OSC"
        hdr.append(('CAMERA'   , value                                            , "Type of camera used (Mono/OSC)"), end=True)
        
        # wavelength units
        if exportAsAngstrom: 
            value = "Angstrom"
            exportFactor = 10
        else:
            value = "nanometre"
            exportFactor = 1
        hdr.append(('CUNIT1'   , value                                            , "Unit of wavelength axis"), end=True)
        
        # calibration details
        if self.isCalibrated:
            value = "Calibrated"
        else:
            value = "No"
        hdr.append(('CALIBRAT' , value                                            , "Calibration method"), end=True)
        hdr.append(('CAL_LAMP' , CALIBRATIONLAMPS[self.calibrationLampId][0]      , "Calibration lamp"), end=True)
        hdr.append(('CAL_PTS'  , len(self.calibrationResidualPlotData)            , "Number of calibration points"), end=True)
        value = self.calibrationResidualPlotData[0][0]*exportFactor
        hdr.append(('CAL_RANL' , value                                            , "Lower limit calibration range in CUNIT1"), end=True)
        value = self.calibrationResidualPlotData[len(self.calibrationResidualPlotData)-1][0]*exportFactor
        hdr.append(('CAL_RANU' , value                                            , "Upper limit calibration range in CUNIT1"), end=True)
        value = float("{:.5f}".format(self.dispersion*exportFactor))
        hdr.append(('DISPERSI' , value                                            , "Dispersion in CUNIT1 per pixel"), end=True)
        value = float("{:.8f}".format(self.R_sq))
        hdr.append(('CAL_Rsq'  , value                                            , "R-squared"), end=True)
        if self.calibrationRequested:
            value = float("{:.8f}".format(self.chiSquaredPdoF))
        else:
            value = "See previous run"
        hdr.append(('CAL_Xsq'  , value                                            , "Chi-Squared per Degree of Freedom"), end=True)
        
        # filter/smoothing settings
        if self.runningAvgAmount > 1:
            value = "Running average stack"
        else:
            value = "Single exposure"
        hdr.append(('INTEGRAT' , value                                            , "Integration mode"), end=True)
        hdr.append(('NUMFRAME' , self.runningAvgAmount                            , "Number of exposures in this integration"), end=True)
        hdr.append(('NUMROWS'  , self.sampleHeight                                , "Number of pixel rows in this integration"), end=True)
        if self.useSavitzkyGolayFilter:
            value = "Savitzky Golay Filter"
        else:
            value = "None"
        hdr.append(('FILTER'   , value                                            , "Additional filtering"), end=True)
        if self.useSavitzkyGolayFilter:
            hdr.append(('FILT_VAL'   , self.savgolFilterPolynomial                    , "Additional filter setting"), end=True)
        
        # darks
        if self.applyDarks:
            value = "Yes"
        else:
            value = "No"
        hdr.append(('DARKS'    , value                                            , "Darks applied?"), end=True)
        
        
        # normalisation
        if self.showNormalised:
            value = "Regular"
        else:
            value = "None"
        hdr.append(('NORMMODE' , value                                            , "Normalisation method"), end=True)
        
        # exposure and gain
        hdr.append(('EXPTIME'  , int((self.runningAvgAmount*self.cameraExposure)) , "Integration time in milliseconds"), end=True)
        hdr.append(('GAIN'     , self.cameraGain                                  , "Camera gain"), end=True)
        
        # software
        hdr.append(('SOFTWARE' , PROGRAMNAME                                      , "Software used to create this 1D-profile"), end=True)
        hdr.append(('VERSION'  , PROGRAMVERSION                                   , "Software version"), end=True)
        hdul.flush()
        hdul.close()
        
        '''
        # we could export as CSV as well to allow comparion with raw data
        print("writing FITS csv")
        f = open("./data/Spectrum-"+now+'-1D_FITS.csv','w')
        f.write('Wavelength,Intensity\n')
        for x in zip(wvlEquidist,intEquidist):
            f.write(str(x[0])+','+str(x[1])+'\n')
        f.close()
        '''

    # ------------------------------------------------------------------------------------
    def SetCalibrationStatus(self):
        theStatus = "Status: "
        if self.isCalibrated:
            theStatus += "C."

        if self.applyDarks:
            theStatus += "D."

        if self.showNormalised:
            theStatus += "N."
        
        self.midPanTop.calibrateStatusLabel.SetLabel(theStatus)
        self.midPanTop.polyFitLabel.SetLabel("Fit: "+POLYFITS[self.polyfitDegree-1])
        self.midPanTop.dispersionLabel.SetLabel("Dispersion: {:.2f}nm/px".format(self.dispersion))
        if self.calibrationRequested:
            self.midPanTop.qualityLabel.SetLabel("\N{GREEK SMALL LETTER CHI}"+chr(0xB2)+": {:.2f}".format(self.chiSquaredPdoF))
        else:
            self.midPanTop.qualityLabel.SetLabel("R"+chr(0xB2)+": {:.7f}".format(self.R_sq))

    # ------------------------------------------------------------------------------------
    def Calibrate(self,evt):
        if (self.showNormalised):
            dial = wx.MessageDialog ( self, 'Calibration cannot be done when normalised', 'Notification', wx.OK)
            dial.ShowModal()
        else:
            self.calibrationRequested = True
            if self.isCalibrated:
                dial = wx.MessageDialog ( self, 'System already calibrated\nAre you sure you want to recalibrate?', 'Question', wx.YES_NO | wx.ICON_EXCLAMATION )
                if dial.ShowModal() == wx.ID_YES:
                   self.AutoCalibrate()
            else:
                self.AutoCalibrate()
            self.calibrationRequested = False


    # ------------------------------------------------------------------------------------
    def ReadCalibrationFile(self):
        #read in the calibration points
        #compute second or third order polynimial, and generate wavelength array!
        #Nicolàs de Hilster, 27 December 2024
        #based on Les Wright 28 Sept 2022

        errors = 0
        try:
            print("Loading calibration data...")
            file = open('caldata.txt', 'r')
        except:
            errors = 1

        try:
            #read both the pixel numbers and wavelengths into two arrays.
            lines = file.readlines()
            line0 = lines[0].strip() #strip newline
            pixels = line0.split(',') #split on ,
            self.originalCalibrationPixels = [float(i) for i in pixels] #convert list of strings to ints
            line1 = lines[1].strip()
            wavelengths = line1.split(',')
            self.correspondingWavelengths = [float(i) for i in wavelengths]#convert list of strings to floats
        except:
            errors = 1

        try:
            if (len(self.originalCalibrationPixels) != len(self.correspondingWavelengths)):
                #The Calibration points are of unequal length!
                errors = 1
            if (len(self.originalCalibrationPixels) < 3):
                #The Cal data contains less than 3 pixels!
                errors = 1
            if (len(self.correspondingWavelengths) < 3):
                #The Cal data contains less than 3 wavelengths!
                errors = 1
        except:
            errors = 1

        if errors == 1:
            print("Loading of Calibration data failed (missing caldata.txt or corrupted data!")
            print("Loading placeholder data...")
            print("You MUST perform a Calibration to use this software!\n\n")
            self.originalCalibrationPixels = [0,400,800]
            self.correspondingWavelengths = [380,560,750]
        
        # calculate the coefficients if we have enough data points
        if len(self.correspondingWavelengths) > 3:
            self.CalculateCalibrationCoefficients()
        # otherwise calibration is not possible and we generate wavelengthData to satisfy the program
        else:
            self.wavelengthData = [0] * self.cameraWidth # 0nm for each pixel
        
    
    # ------------------------------------------------------------------------------------
    def AutoCalibrate(self):
                
        
        print("Calibrating using "+CALIBRATIONLAMPS[self.calibrationLampId][0])
        calibrationWavelengths = CALIBRATIONLAMPS[self.calibrationLampId][1]
#        self.calibrationFirstPeak = CALIBRATIONLAMPS[self.calibrationLampId][2]
#        self.calibrationLastPeak = CALIBRATIONLAMPS[self.calibrationLampId][3]
#        self.calibrationDetectionLevel = CALIBRATIONLAMPS[self.calibrationLampId][4]
                
        # recalculate the peaks with a lower peak distance to ensure all peaks are found
        pt = int(self.peakThreshold) #make sure the data is int.
        pixels = peakIndexes(self.intensity, thres=pt/max(self.intensity), min_dist=5)

        maxInt = max(self.intensity)
        firstPeakPx = 0 # px
        lastPeakPx = 0 # px
        
        # auto calibrate uses the first and last peak that is more than the set percentage of the maximum peak
        firstPeakWl = calibrationWavelengths[self.calibrationFirstPeak]
        lastPeakWl = calibrationWavelengths[self.calibrationLastPeak]
        
        estimatedWavelengths = [] # first estimation of wavelengths
        self.correspondingWavelengths = [] # the actual wavelengths
        self.correspondingWavelengthDifferences = []
        self.originalCalibrationPixels = [] # copy of pixels array as list
        if VERBOSE:
            print ("################################# START CALIBRATION #############################")
            print ("Estimating peak wavelengths based on significant peaks at {:.2f}nm and {:.2f}nm".format(firstPeakWl, lastPeakWl))
            print ("")
        
        # first find peaks belonging to 435.833nm and 611.00nm
        if VERBOSE:
            print ("Peaks detected at:")
        idx = 0
        for px in pixels:
            str1 = ""
            str2 = ""
            if (self.intensity[px] > maxInt * self.calibrationDetectionLevel) and (firstPeakPx == 0):
                str1 = "FP" # First Peak
                firstPeakPx = px
            if (self.intensity[px] > maxInt * self.calibrationDetectionLevel):
                str2 = "NP" # Next Peak
                lastPeakPx = px
            if VERBOSE:
                print ("{}: {}px = {}ADU {} {}".format(idx,px,self.intensity[px],str2,str1))
            idx += 1
        
        # factor to go from pixels to nanometers
        factor = (lastPeakWl - firstPeakWl) / (lastPeakPx-firstPeakPx)
        if VERBOSE:
            print ("")
            print ("factor: ",factor)
            print ("")

        # first stage: find the approximate wavelengths for all detected peaks, but this will result in double hits 
        idx = 0
        if VERBOSE:
            print ("Estimated wavelengths")
        for px in pixels:
            ewl = firstPeakWl + (px - firstPeakPx) * factor
            peakDiff = 9999
            for cwl in calibrationWavelengths:
                diff = abs(ewl - cwl)
                if diff < peakDiff:
                    peakWl = cwl
                    peakDiff = diff
            self.originalCalibrationPixels.append(px)
            estimatedWavelengths.append(ewl)
            self.correspondingWavelengths.append(peakWl)
            if VERBOSE:
                print ("{}: peak @ {}px: estimated = {:.1f}nm, actual = {:.1f}nm, difference = {:.1f}nm".format(idx, px, ewl, peakWl, ewl-peakWl))
            idx += 1
 
        # second stage: create a list of false detections (these are double entries, so we want to keep those that are nearest to the actual wavelength)
        idx = 0
        oldCWL = 0
        delIdx = []
        oldDiff = 0
        for cwl in self.correspondingWavelengths:
            diff = abs(estimatedWavelengths[idx]-cwl)
            # check if entry is double or difference too large
            if cwl == oldCWL or diff > (self.calibrationRejectionLevel+2*self.calibrationSpectrumQuality):
                if diff < oldDiff:
                    delIdx.append(idx-1)
                else:
                    delIdx.append(idx)
            oldCWL = cwl
            oldDiff = diff
            idx += 1
        
        if VERBOSE:
            print ("Removing double or poor estimated entries:",delIdx)
        
        # third stage: delete the double entries
        idx = 0
        oldDidx = -1
        for didx in delIdx:
            if didx != oldDidx:
                if VERBOSE:
                    print("Removing ", didx)
                self.correspondingWavelengths.pop(didx-idx)
                estimatedWavelengths.pop(didx-idx)
                self.originalCalibrationPixels.pop(didx-idx)
                idx += 1
            oldDidx = didx

        # show what is left
        if VERBOSE:
            idx = 0
            print ("Remaining wavelengths:")
            for px in self.originalCalibrationPixels:
                print ("{}: peak @ {}px: estimated = {:.1f}nm, actual = {:.1f}nm, difference = {:.1f}nm".format(idx, px, estimatedWavelengths[idx], self.correspondingWavelengths[idx], estimatedWavelengths[idx]-self.correspondingWavelengths[idx]))
                idx += 1


        # calculate the coefficients if we have enough data points
        if len(self.correspondingWavelengths) > 3:
            self.CalculateCalibrationCoefficients()
        # otherwise calibration is not possible and we generate wavelengthData to satisfy the program
        else:
            self.wavelengthData = [0] * self.cameraWidth # 0nm for each pixel

    # ------------------------------------------------------------------------------------
    def CalculateCalibrationCoefficients(self):

        print("Calibrated using "+CALIBRATIONLAMPS[self.calibrationLampId][0])
        # we going to do iterations to remove outliers
        iteration = 0
        outliersFound = True

        cleanedUpCalibrationPixels = self.originalCalibrationPixels
        cleanedUpCorrespondingWavelengths = self.correspondingWavelengths

        while outliersFound:
            # now calculate the coefficients
            # NHI changed to 2nd degree order by default and added linear fit (1st degree polynome) and 4th degree as additional options
            print ("Iteration ",iteration)
            coefficients = np.poly1d(np.polyfit(cleanedUpCalibrationPixels, cleanedUpCorrespondingWavelengths, self.polyfitDegree))
            coefficientsLin = np.poly1d(np.polyfit(cleanedUpCalibrationPixels, cleanedUpCorrespondingWavelengths, 1))

            if VERBOSE:
                print("")
                print("Creating best-fit polynomal")
                print(coefficients)
            
            # create table of wavelengths for each camera pixel
            if VERBOSE:
                print("")
                print("Generating Wavelength Data for {} pixels!\n\n".format(self.cameraWidth))
        
            self.wavelengthData = []
            self.calibrationResidualPlotData = []
            self.calibrationResidualPlotDataLinear = []
            self.calibrationPolynomeDifferenceWithLinear = []
            cPDWLTemp = []
            for px in range(self.cameraWidth):
                y = 0
                y_linear = 0
                for n in range(len(coefficients)+1):
                    y += coefficients[n] * float(px)**n       
                    if (n<=1):
                        y_linear += coefficientsLin[n] * float(px)**n       
                wavelength = y # ((C1*px**3)+(C2*px**2)+(C3*px)+C4)
                wavelength = round(wavelength,6) #because seriously!
#                print(n,px,y,wavelength)
                self.wavelengthData.append(wavelength)
                cPDWLTemp = round(y_linear,6)
                self.calibrationPolynomeDifferenceWithLinear.append((wavelength,wavelength-cPDWLTemp))
#                print(wavelength,cPDWLTemp,wavelength-cPDWLTemp)
        
        
            # final job, we need to compare all the recorded wavelengths with predicted wavelengths
            predicted = []
            # iterate over the original pixelnumber array and predict results
            idx = 0
            print ("Calculated wavelengths for peaks using "+POLYFITS[self.polyfitDegree-1]+" best-fit:")
            sumSquaredResiduals = 0
            variances = []
            self.updatedCalibrationPixels = []
            self.correspondingWavelengthDifferences = []
            if iteration == 0:
                variancesTmp = []
            for px in cleanedUpCalibrationPixels:
                # when doing a new calibration we want to calculate chi-squared
                # as the peak finding routine does only return the exact pixel where the peak occurs we do a Gaussian fit to see how accurately the pixel position could be found
                # but only in the first iteration
                if iteration == 0:
                    orgPx = px
                    if self.calibrationRequested:
                        # find the range of the peak by looking where the slope ends on both sides of the peak
                        iidx = 1
                        prevIntensity = self.intensityFloat[px]
                        while self.intensityFloat[px-iidx] <= prevIntensity and self.intensityFloat[px-iidx] >= self.intensityFloat[px]*0.1:
                            prevIntensity = self.intensityFloat[px-iidx]
                            iidx += 1
                        self.FWHM_RangeStart = px - iidx + 1
                        iidx = 1
                        prevIntensity = self.intensityFloat[px]
                        while self.intensityFloat[px+iidx] <= prevIntensity and self.intensityFloat[px+iidx] >= self.intensityFloat[px]*0.1:
                            prevIntensity = self.intensityFloat[px+iidx]
                            iidx += 1
                        self.FWHM_RangeEnd = px + iidx - 1
#                        print("testje",self.FWHM_RangeStart,px,self.FWHM_RangeEnd)
                        if (self.FWHM_RangeEnd-self.FWHM_RangeStart)>4:
                            try:
                                # then calculate Gaussian best fit...
                                x_crop, bestFit, cov = self.CalculateGaussianFWHM()
                                '''
                                print(cov)
                                sigmas = np.sqrt(np.diag(cov))
                                print("Sqrt of diagonal: ",sigmas,sigmas[1],sigmas[1]**2)
                                '''
                                # and get the accuracy of the peak position from the covariance matrix
                                try:
                                    variancesTmp.append(min(float(cov[1][1]),999.0))
                                except:
                                    variancesTmp.append(0.0)
                                # get the float pixel location of the maximum
                                # disabled it as the asymmetrical shape of the Fluorescent light bulb peaks are not Gaussian
#                                print(idx,px,bestFit[1])
                                if CALIBRATIONLAMPS[self.calibrationLampId][5]:
                                    px = bestFit[1]
                                    self.originalCalibrationPixels[idx] = px # otherwise we see the difference between the Gaussian fit px-value and the final derived result
                            except:
                                pass
                        else:
                            variancesTmp.append(999.0)
                    else:
                        variancesTmp.append(999.0)
                
                self.updatedCalibrationPixels.append(px)
        
                y = 0
                y_linear = 0
                for n in range(len(coefficients)+1):
                    y += (coefficients[n] * float(px)**n) # the float(px) is required to avoid an overflow at orders 3 and 4
                    if (n<=1):
                        y_linear += coefficientsLin[n] * float(px)**n       
#                    print(n, coefficients[n],y)
                print ("{}: peak @ {}->{:.1f}px: theoretical = {:.3f}nm, calculated = {:.3f}nm, difference = {:.3f}nm".format(idx,orgPx,px,cleanedUpCorrespondingWavelengths[idx],y,cleanedUpCorrespondingWavelengths[idx]-y))
                predicted.append(y)
                self.calibrationResidualPlotDataLinear.append((cleanedUpCorrespondingWavelengths[idx],cleanedUpCorrespondingWavelengths[idx]-y_linear))
                self.calibrationResidualPlotData.append((cleanedUpCorrespondingWavelengths[idx],cleanedUpCorrespondingWavelengths[idx]-y))
                self.correspondingWavelengthDifferences.append(cleanedUpCorrespondingWavelengths[idx]-y)
                sumSquaredResiduals += (cleanedUpCorrespondingWavelengths[idx]-y)**2
        
                idx += 1
            # we need to know the dispersion in nm/px
            self.dispersion = (cleanedUpCorrespondingWavelengths[len(self.updatedCalibrationPixels)-1]-cleanedUpCorrespondingWavelengths[0])/(self.updatedCalibrationPixels[len(self.updatedCalibrationPixels)-1]-self.updatedCalibrationPixels[0])
            print("Dispersion for {}nm ({}px) to {}nm ({}px): {}nm/px".format(cleanedUpCorrespondingWavelengths[0], self.updatedCalibrationPixels[0], cleanedUpCorrespondingWavelengths[len(self.updatedCalibrationPixels)-1], self.updatedCalibrationPixels[len(self.updatedCalibrationPixels)-1], self.dispersion))
#            print("Pixel variances: ",variancesTmp,"px")
            for v in variancesTmp:
                variances.append(v * self.dispersion)
#            print("Wavelength variances: ",variances,"nm")
            if self.calibrationRequested:
                self.chiSquaredPdoF = self.ChiSquaredPerDegreeOfFreedom(predicted,cleanedUpCorrespondingWavelengths,variancesTmp) # https://en.wikipedia.org/wiki/Reduced_chi-squared_statistic
            print("Chi-squared RSS: ",sumSquaredResiduals/(len(self.updatedCalibrationPixels)-1)) # https://en.wikipedia.org/wiki/Reduced_chi-squared_statistic
            # calculate 2 squared of the result
            # if this is close to 1 we are all good!
            corr_matrix = np.corrcoef(cleanedUpCorrespondingWavelengths, predicted)
            corr = corr_matrix[0,1]
            self.R_sq = corr**2
            
            print("R-Squared = {:.7f}".format(self.R_sq))
            if self.calibrationRequested:
                print("Chi-Squared per degree of freedom = {:.3f}".format(self.chiSquaredPdoF))
            else:
                print("Chi-Squared per Degree of Freedom can only be calculated with live data")
            print()
            
            # now detect and remove outliers using MAD-filter (MAD = Medium Absolute Deviation)
            outliersFound = False
            theMedian = np.median(self.correspondingWavelengthDifferences)
            theABSDiffWithMedian = abs(self.correspondingWavelengthDifferences-theMedian)
            theMAD = np.median(theABSDiffWithMedian)*1.4826
            numPx = len(cleanedUpCalibrationPixels)
            tmpPixels = []
            tmpWavelengths = []
            for inp in range (numPx):
                if theABSDiffWithMedian[inp] <= theMAD*2:
                    tmpPixels.append(cleanedUpCalibrationPixels[inp])
                    tmpWavelengths.append(cleanedUpCorrespondingWavelengths[inp])
                else:
                    print("Outlier found at spectral line {}: {:.3f}".format(inp,cleanedUpCorrespondingWavelengths[inp]))
                    outliersFound = True
#                print(theABSDiffWithMedian[inp])
            cleanedUpCalibrationPixels = tmpPixels
            cleanedUpCorrespondingWavelengths = tmpWavelengths
#            print(theMedian)
#            print(theABSDiffWithMedian)
#            print(theMAD)
            iteration += 1
            # exit if too few spectral lines left for a reliable solution or after too may iterations
            if numPx <= (self.polyfitDegree+2) or iteration > 10:
                outliersFound = False
        

        
        # finally save the new calibration:
        pxdata = ','.join(map(str, cleanedUpCalibrationPixels)) #convert array to string
        wldata = ','.join(map(str, cleanedUpCorrespondingWavelengths)) #convert array to string
        f = open('caldata.txt','w')
        f.write(pxdata+'\n')
        f.write(wldata+'\n')
        self.isCalibrated = True
        self.calibrationStartTime = time.time()
        self.toolbar.EnableTool(ID_SCL, self.isCalibrated)
        self.SetCalibrationStatus()
        
        # now make an array with all spectra lines found before filtering to display them as well in the plot
        calibrationResidualPlotDataLinearRaw = []
        idx = 0
        for px in self.originalCalibrationPixels:
            y_linear = 0
            for n in range(len(coefficientsLin)+1):
                y_linear += coefficientsLin[n] * float(px)**n       
            calibrationResidualPlotDataLinearRaw.append((self.correspondingWavelengths[idx],self.correspondingWavelengths[idx]-y_linear))
            idx += 1

        print ("Calibration solution found after {} iterations".format(iteration-1))
        print ("Calibration points: ",numPx)
        print ("Rejected points: ", idx-numPx)

        # and show it in the residual plot window
        if self.calibrationResidualPlotFrameIsOpen:
            self.calibrationResidualPlotFrame.plotData(calibrationResidualPlotDataLinearRaw,self.calibrationResidualPlotDataLinear,self.calibrationResidualPlotData,self.calibrationPolynomeDifferenceWithLinear)

    # ------------------------------------------------------------------------------------
    def ChiSquaredPerDegreeOfFreedom(self,calculated,observed,variances):
        
        i = 0
        u = 0
        chiSquared = 0
        for c in calculated:
            if variances[i] < 1.0:
                chiSquared += ((observed[i]-calculated[i])**2)/variances[i] # https://en.wikipedia.org/wiki/Reduced_chi-squared_statistic
                u += 1
            i += 1
        chiSquaredPerDegreeOfFreedom = chiSquared/(u-1)
        return chiSquaredPerDegreeOfFreedom

    # ------------------------------------------------------------------------------------
    def ShowCalibrationLines(self,evt):
        self.calibrationStartTime = time.time()
      
    # ------------------------------------------------------------------------------------
    def TakeDarks(self,evt):

        if (self.showNormalised):
            dial = wx.MessageDialog ( self, 'Darks cannot be taken or switched off when normalised', 'Notification', wx.OK)
            dial.ShowModal()
        else:
            if (self.applyDarks): # we want to turn off darks
                dial = wx.MessageDialog ( self, 'Are you sure you want to turn off the darks?', 'Question', wx.YES_NO | wx.ICON_EXCLAMATION )
                if dial.ShowModal() == wx.ID_YES:
                    self.darkData = [0] * len(self.intensity) #array for dark data...full of zeros
                    self.applyDarks = False
            else: # we want to turn normalising ON
                self.darkData = self.intensityFloat.copy()
                self.applyDarks = True
    
            self.SetCalibrationStatus()

    # ------------------------------------------------------------------------------------
    def CropControl(self,evt,updown):
        self.sampleHeight += updown
        if self.sampleHeight >= self.sampleHeightMax:
            self.sampleHeight = self.sampleHeightMax
        if self.sampleHeight <= 0:
            self.sampleHeight = 1
        self.shift_down = False
        self.midPanTop.sampleHeightLabel.SetLabel("Sample height: "+str(self.sampleHeight)+" rows")

    # ------------------------------------------------------------------------------------
    def FilterControl(self,evt,updown):
        if self.shift_down:
            self.runningAvgAmount += updown * 10
        else:
            self.runningAvgAmount += updown
        if self.runningAvgAmount <= 0:
            self.runningAvgAmount = 1
        self.shift_down = False
        if self.runningAvgCounter > self.runningAvgAmount:
            self.runningAvgCounter = self.runningAvgAmount
        self.midPanTop.runningAvgLabel.SetLabel("Running AVG: {}/{}".format(self.runningAvgCounter,self.runningAvgAmount))

    # ------------------------------------------------------------------------------------
    def GainControl(self,evt,updown):

        if self.canSetGain:
            if self.shift_down:
                self.cameraGain += updown * 10
            else:
                self.cameraGain += updown
            if self.cameraGain <= 0:
                self.cameraGain = 1
            elif self.cameraGain > self.cameraMaxGain:
                self.cameraGain = self.cameraMaxGain
    
            if self.cameraIsPi:
                self.picam2.set_controls({"AnalogueGain": self.cameraGain})
            if self.cameraIsASI:
                self.ASI_camera.set_control_value(asi.ASI_GAIN, self.cameraGain)
        
        self.shift_down = False
        self.midPanTop.cameraGainLabel.SetLabel("Gain: {}/{}".format(self.cameraGain,self.cameraMaxGain))
        evt.Skip()
    
    # ------------------------------------------------------------------------------------
    def ExposureControl(self,evt,updown):

        if self.canSetExposure:
            if (updown == 1):
                if self.shift_down:
                    self.cameraExposure = int(self.cameraExposure * 1.10)
                else:
                    if self.cameraExposure > 100:
                        self.cameraExposure = int(self.cameraExposure * 1.01)
                    else:
                        self.cameraExposure += 1
            else:
                if self.shift_down:
                    self.cameraExposure = int(self.cameraExposure * 0.90)
                else:
                    self.cameraExposure = int(self.cameraExposure * 0.99)
            if self.cameraExposure <= 1:
                self.cameraExposure = 1
            elif self.cameraExposure > self.cameraMaxExposure:
                self.cameraExposure = self.cameraMaxExposure
            
            if self.cameraIsASI:
                self.ASI_camera.set_control_value(asi.ASI_EXPOSURE, self.cameraExposure*1000)
        
        self.shift_down = False
        self.midPanTop.cameraExposureLabel.SetLabel("Exposure: {}/{}ms".format(self.cameraExposure,self.cameraMaxExposure))
        evt.Skip()

    # ------------------------------------------------------------------------------------
    def SetCursorPosition(self, evt):

        selectionStartPos = self.dataIdx
    
    # ------------------------------------------------------------------------------------
    def GetPeak(self, evt):

        selectionEndPos = self.dataIdx
        maxVal = 0
        maxValPos = 0
        for i in range (selectionStartPos, selectionEndPos):
            curVal = self.intensity[i]
            if curVal > maxVal:
                maxVal = curVal
                maxValPos = i
        
    # ------------------------------------------------------------------------------------
    def GetDataIdx(self,evt):

        # here we only register the mouse-click locations, toggling the FWHM function on/off is done in self.ToggleMeasureFWHM(), calculating the FWHM in self.calculateFWHM()
        if self.doFWHM_Measurement:
            # first determine the start of the range
            if not self.FWHM_MeasurementLeftDone:
                self.FWHM_RangeStart = self.dataIdx
                self.FWHM_MeasurementLeftDone = True
                self.FWHM_MeasurementRightDone = False

            # then the end of the range
            elif not self.FWHM_MeasurementRightDone:
                self.FWHM_RangeEnd = self.dataIdx
                # check if user first clicked right and then left, if so, reverse the positions
                if self.FWHM_RangeEnd < self.FWHM_RangeStart:
                    self.FWHM_RangeEnd = self.FWHM_RangeStart
                    self.FWHM_RangeStart = self.dataIdx
                # check if the user selected a valid range to detect a peak (minimum 3 pixels), else render the selection void
                if abs(self.FWHM_RangeEnd - self.FWHM_RangeStart) >= 3:
                    self.FWHM_MeasurementRightDone = True
                    self.FWHM_MeasurementLeftDone = False
                else:
                    self.FWHM_MeasurementRightDone = False
                    self.FWHM_MeasurementLeftDone = False
        else:
            if (self.showNormalised):
                print("Data at cursor ({0}): Lambda = {1:.2f}nm, i = {2:.2f}".format(self.dataIdx,self.wavelengthData[self.dataIdx], self.intensity[self.dataIdx]))
            else:
                print("Data at cursor ({0}): Lambda = {1:.2f}nm, i = {2:0}".format(self.dataIdx,self.wavelengthData[self.dataIdx], self.intensity[self.dataIdx]))

    # ------------------------------------------------------------------------------------
    def OnKeyPress(self, evt):
    
        keycode = evt.GetKeyCode()
        self.shift_down = False
        if keycode == wx.WXK_SHIFT: # is shift pressed for fast increase/decrease
            self.shift_down = True
        elif keycode == ord(ID_DAR_KEY): # Toggle Darks On/Off
            self.TakeDarks(evt)
        elif keycode == ord(ID_GPL_KEY): # Gain UP
            if self.canSetGain:
                self.GainControl(evt,1)
        elif keycode == ord(ID_GMI_KEY): # Gain DOWN
            if self.canSetGain:
                self.GainControl(evt,-1)
        elif keycode == ord(ID_EPL_KEY): # Exposure UP
            if self.canSetExposure:
                self.ExposureControl(evt,1)
        elif keycode == ord(ID_EMI_KEY): # Exposure DOWN
            if self.canSetExposure:
                self.ExposureControl(evt,-1)
        elif keycode == ord(ID_HPE_KEY): # Toggle Hold Peaks On/Off
            self.ToggleHoldPeaks(evt)
        elif keycode == ord(ID_NOR_KEY): # Normalising
            self.ToggleNormalisation(evt)
        elif keycode == ord(ID_CAL_KEY): # Calibrate
            self.Calibrate(evt)
        elif keycode == ord(ID_CMI_KEY): # Crop height min
            self.CropControl(evt, -1)
        elif keycode == ord(ID_CPL_KEY): # Crop height plus
            self.CropControl(evt, 1)
        elif keycode == ord(ID_FPL_KEY): # Running average UP
            self.FilterControl(evt,1)
        elif keycode == ord(ID_FMI_KEY): # Running average DOWN
            self.FilterControl(evt,-1)
        elif keycode == ord(ID_SGT_KEY): # Toggle Savitzky-Golay filter ON/OFF
            self.ToggleSavitzkyGolayFilter(evt, 0)
        elif keycode == ord(ID_SGP_KEY): # Increase Savitzky-Golay filter
            self.ToggleSavitzkyGolayFilter(evt, 1)
        elif keycode == ord(ID_SAV_KEY): # Save the files
            self.Save(evt)
        elif keycode == ord(ID_SGT_KEY): # Toggle Savitzky-Golay Filter ON/OFF
            self.ToggleSavitzkyGolayFilter(evt, 0)
        elif keycode == ord(ID_SGP_KEY): # Increase Savitzky-Golay Filter
            self.ToggleSavitzkyGolayFilter(evt, 1)
        elif keycode == ord(ID_MEG_KEY): # Toggle Measure Gaussian FWHM ON/OFF
            self.ToggleMeasureFWHM(evt,FWHM_GAUSSIAN)
        elif keycode == ord(ID_MEB_KEY): # Toggle Measure Block FWHM ON/OFF
            self.ToggleMeasureFWHM(evt,FWHM_BLOCK)
        elif keycode == ord(ID_SCL_KEY): # Show the calibration lines
            self.ShowCalibrationLines(evt)
        elif keycode == ord(ID_SRP_KEY): # Show the Residual Plot
            self.ToggleResidualPlot(evt)
        elif keycode == ord(ID_CVE_KEY): # Recentre the spectrum vertically
            self.RecentreSpectrum(evt)
        elif keycode == ord(ID_CLA_KEY): # Show the calibration lamp set-up
            self.showCalibrationLampSetUpFrame(evt)
        elif keycode == ord(ID_SCP_KEY): # Show the camera preview
            self.ToggleCameraPreviewFrame(evt)
        elif keycode == ord(ID_RES_KEY): # Reset running average (restarts sampling from the last frame)
            self.ResetRunningAverage(evt)
        
        evt.Skip()


    # ------------------------------------------------------------------------------------
    def OnMouseMove(self, evt):
        w, h = self.midPanBottom.GetSize()
        ctrl_pos = evt.GetPosition()
#            print("ctrl_pos: " + str(ctrl_pos.x) + ", " + str(ctrl_pos.y))
#            pos = self.midPanBottom.ScreenToClient(ctrl_pos)
#            print ("pos relative to screen top left = " + str(pos))
#            screen_pos = self.GetScreenPosition()
#            relative_pos_x = pos[0] + screen_pos[0]
#            relative_pos_y = pos[1] + screen_pos[1]
#            print ("pos relative to image top left = " + str(relative_pos_x) + " " + str(relative_pos_y))
        self.dataIdx = int(ctrl_pos.x/w*self.cameraWidth)

        if (self.showNormalised):
            self.statusbar.SetStatusText("Px {0}, Lambda = {1:.2f}nm, i = {2:.0f}%".format(self.dataIdx,self.wavelengthData[self.dataIdx], self.intensity[self.dataIdx]*100))
        else:
            self.statusbar.SetStatusText("Px {0}, Lambda = {1:.2f}nm, i = {2:0.0f}ADU".format(self.dataIdx,self.wavelengthData[self.dataIdx], self.intensity[self.dataIdx]))


    # ------------------------------------------------------------------------------------
    def SavePrefs(self):
        f = open(self.thePrefFileName,'w')
        f.write('[Calibration Lamp Id]\n')
        f.write(str(self.calibrationLampId)+'\n')
        f.write('[Calibration First Peak]\n')
        f.write(str(self.calibrationFirstPeak)+'\n')
        f.write('[Calibration Last Peak]\n')
        f.write(str(self.calibrationLastPeak)+'\n')
        f.write('[Calibration Detection Level]\n')
        f.write(str(self.calibrationDetectionLevel)+'\n')
        f.write('[Calibration Rejection Level]\n')
        f.write(str(self.calibrationRejectionLevel)+'\n')
        f.write('[Polyfit Degree]\n')
        f.write(str(self.polyfitDegree)+'\n')
        f.write('[Sample Height]\n')
        f.write(str(self.sampleHeight)+'\n')
        f.write('[Running average filter]\n')
        f.write(str(self.runningAvgAmount)+'\n')
        f.write('[Savitzky-Golay filter enabled]\n')
        f.write(str(self.useSavitzkyGolayFilter)+'\n')
        f.write('[Savitzky-Golay filter value]\n')
        f.write(str(self.savgolFilterPolynomial)+'\n')
        f.write('[Hold peaks]\n')
        f.write(str(self.holdPeaks)+'\n')
        f.write('[Gain]\n')
        f.write(str(self.cameraGain)+'\n')
        f.write('[Exposure]\n')
        f.write(str(self.cameraExposure)+'\n')
        f.write('[Spectrum Vertical Offset]\n')
        f.write(str(self.verticalOffset)+'\n')
        if self.windowIsMaximized: # we remember the minimized position
            # as there is no way to bind the minimize-button, we need to check whether the window is still maximized, a status we prefer not to remember
            # and as DisplaySize is a few pixels (but how many?) larger than self.GetSize, we do an estimation based on the 95% screen-diagonal
            wS,hS = self.GetSize()
            wD,hD = wx.DisplaySize()
            if math.sqrt(wS**2+hS**2) > math.sqrt(wD**2+hD**2)*.95: # it still is more or less maximized and we store the last recorded minimized position (and size further below)
                self.windowIsMaximized = True
            else: # we say it is no longer maximized to store the current window size further below
                self.windowIsMaximized = False
            x, y = (self.windowPosX,self.windowPosY)
        else:
            x, y = self.GetPosition()
        f.write('[Window position X]\n')
        f.write(str(x)+'\n')
        f.write('[Window position Y]\n')
        f.write(str(y)+'\n')
        if self.windowIsMaximized: # we remember the minimized size
            x, y = (self.windowSizeX,self.windowSizeY)
        else:
            x, y = self.GetSize()
        f.write('[Window size X]\n')
        f.write(str(x)+'\n')
        f.write('[Window size Y]\n')
        f.write(str(y)+'\n')
        if self.applyDarks:
            f.write('[Dark]\n')
            for d in self.darkData:
                f.write(str(d)+',')
        f.close()

    # ------------------------------------------------------------------------------------
    def ReadPrefs(self):
        if VERBOSE:
            print ("Reading last used settings from "+self.thePrefFileName)
        try:
            with open(self.thePrefFileName,'r') as prefsFile:
                while True:
                    line = prefsFile.readline()
                    if not line:
                        break
                    data = prefsFile.readline()
                    if not data:
                        break
                    line = line.strip()
                    data = data.strip()

                    if line == '[Calibration Lamp Id]':
                        self.calibrationLampId = int(data)
                        if VERBOSE:
                            print ("Calibration Lamp Id: ", self.calibrationLampId)
                    if line == '[Calibration First Peak]':
                        self.calibrationFirstPeak = int(data)
                        if VERBOSE:
                            print ("Calibration First Peak: ", self.calibrationFirstPeak)
                    if line == '[Calibration Last Peak]':
                        self.calibrationLastPeak = int(data)
                        if VERBOSE:
                            print ("Calibration Last Peak: ", self.calibrationLastPeak)
                    if line == '[Calibration Detection Level]':
                        self.calibrationDetectionLevel = float(data)
                        if VERBOSE:
                            print ("Calibration Detection Level: ", self.calibrationDetectionLevel)
                    if line == '[Calibration Rejection Level]':
                        self.calibrationRejectionLevel = float(data)
                        if VERBOSE:
                            print ("Calibration Rejection Level: ", self.calibrationRejectionLevel)
                    if line == '[Polyfit Degree]':
                        self.polyfitDegree = int(data)
                        if VERBOSE:
                            print ("Polyfit Degree: ", self.polyfitDegree)
                    if line == '[Sample Height]':
                        self.sampleHeight = int(data)
                        if VERBOSE:
                            print ("Sample Height: ", self.sampleHeight)
                    if line == '[Running average filter]':
                        self.runningAvgAmount = int(data)
                        if VERBOSE:
                            print ("Running average filter: ", self.runningAvgAmount)
                    if line == '[Savitzky-Golay filter enabled]':
                        self.useSavitzkyGolayFilter = eval(data)
                        if VERBOSE:
                            print ("Savitzky-Golay filter enabled: ", self.useSavitzkyGolayFilter)
                    if line == '[Savitzky-Golay filter value]':
                        self.savgolFilterPolynomial = int(data)
                        if VERBOSE:
                            print ("Savitzky-Golay filter value: ", self.savgolFilterPolynomial)
                    if line == '[Hold peaks]':
                        self.holdPeaks = eval(data)
                        if VERBOSE:
                            print ("Hold peaks: ", self.holdPeaks)
                    if line == '[Gain]':
                        self.cameraGain = int(data)
                        if VERBOSE:
                            print ("Gain: ", self.cameraGain)
                    if line == '[Exposure]':
                        self.cameraExposure = int(data)
                        if VERBOSE:
                            print ("Exposure: ", self.cameraExposure)
                    if line == '[Spectrum Vertical Offset]':
                        self.verticalOffset = int(data)
                        if VERBOSE:
                            print ("Vertical Offset: ", self.verticalOffset)
                    if line == '[Window position X]':
                        self.windowPosX = int(data)
                        if VERBOSE:
                            print ("Window position X: ", self.windowPosX)
                    if line == '[Window position Y]':
                        self.windowPosY = int(data)
                        if VERBOSE:
                            print ("Window position Y: ", self.windowPosY)
                    if line == '[Window size X]':
                        self.windowSizeX = int(data)
                        if VERBOSE:
                            print ("Window size X: ", self.windowSizeX)
                    if line == '[Window size Y]':
                        self.windowSizeY = int(data)
                        if VERBOSE:
                            print ("Window size Y: ", self.windowSizeY)

                    if line == '[Dark]':
                        splittedData = data[:-1].split(',') #split on ,
                        self.darkData = [float(i) for i in splittedData] 
                        self.applyDarks = True
                        if VERBOSE:
                            print ("Dark read")
            prefsFile.close()
        except FileNotFoundError:
            pass


# ========================================================================================
class ImagePanel(wx.Panel):

    ID_TIMER = 1

    # ------------------------------------------------------------------------------------
    def __init__(self, parent, sb, theSize, theName):
        default = wx.CLIP_CHILDREN | wx.NO_FULL_REPAINT_ON_RESIZE
        super().__init__(parent, style = default)
        self.img = wx.Image(width=1, height=1)
        self.ctrl = wx.StaticBitmap(self, bitmap=self.img.ConvertToBitmap(), size = theSize)
        self.Bind(wx.EVT_SIZE, self.on_resize)
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
        self.ctrl.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
       
        self.bar = sb
#        self.initImg = theImg
        self.theName = theName
        sb.SetStatusText("Init done")

    # ------------------------------------------------------------------------------------
    def update_bitmap(self):
        w, h = self.GetSize()
        # exit this function when empty image
        if w*h == 0:
            return
        self.ctrl.SetBitmap(self.img.Scale(w, h).ConvertToBitmap())

    # ------------------------------------------------------------------------------------
    def on_resize(self, evt):
        self.update_bitmap()
        
    # ------------------------------------------------------------------------------------
    def OnEraseBackground(self, evt):
        # do nothing
        pass

# ========================================================================================
class InfoPanel(wx.Panel):
    # ------------------------------------------------------------------------------------
    def __init__(self, parent, sb, theSize, theName):
        super().__init__(parent)
        self.panelSize = theSize
        self.bar = sb
        self.theName = theName
        #self.InitUI()
        self.SetBackgroundColour("white")
        
        hbox = wx.FlexGridSizer(1, 5, 0, 10)
        # ------ Program name
        self.programNameLabel = wx.StaticText(self, label=parent.Parent.programName)
        self.programNameLabel.SetForegroundColour((255,0,0))
        font = wx.Font(18, wx.DECORATIVE, wx.ITALIC, wx.NORMAL)
        self.programNameLabel.SetFont(font)

        # ------ Camera details
        self.cameraGainLabel = wx.StaticText(self, label="Gain: ---/---")
        self.cameraBitDepthLabel = wx.StaticText(self, label="Bit-depth: --")
        self.cameraCaptureWidthLabel = wx.StaticText(self, label="ImageWidth: ----")
        self.cameraExposureLabel = wx.StaticText(self, label="Exposure: 000/00000ms")
        
        # ------ Calibration details
        self.calibrateStatusLabel = wx.StaticText(self, label="Not calibrated")
        self.polyFitLabel = wx.StaticText(self, label="Not calibrated")
        self.dispersionLabel = wx.StaticText(self, label="Dispersion: --- nm/px")
        self.qualityLabel = wx.StaticText(self, label="R-squared: --------")

        # ------ Filtering
        self.sampleHeightLabel = wx.StaticText(self, label="Sample height: 00px")
        if parent.Parent.useSavitzkyGolayFilter:
            self.savgolFilterLabel = wx.StaticText(self, label="Savgol filter: "+str(parent.Parent.savgolFilterPolynomial))
        else:
            self.savgolFilterLabel = wx.StaticText(self, label="Savgol filter: OFF")
        self.runningAvgLabel = wx.StaticText(self, label="Running AVG: ---")
        self.stabilityLabel = wx.StaticText(self, label="dStDev: 0.00")

        # ------ Miscellaneous
        self.saveDataLabel = wx.StaticText(self, label="No data saved")
        self.holdPeaksLabel = wx.StaticText(self, label="Holdpeaks: OFF")

        self.calibrateStatusLabel.SetForegroundColour((0,0,0))
        self.polyFitLabel.SetForegroundColour((0,0,0))
        self.sampleHeightLabel.SetForegroundColour((0,0,0))
        self.saveDataLabel.SetForegroundColour((0,0,0))
        self.cameraGainLabel.SetForegroundColour((0,0,0))
        self.holdPeaksLabel.SetForegroundColour((0,0,0))
        self.savgolFilterLabel.SetForegroundColour((0,0,0))
        self.runningAvgLabel.SetForegroundColour((0,0,0))
        self.stabilityLabel.SetForegroundColour((0,0,255))
        self.cameraBitDepthLabel.SetForegroundColour((0,0,0))
        self.cameraCaptureWidthLabel.SetForegroundColour((0,0,0))
        self.cameraExposureLabel.SetForegroundColour((0,0,0))
        self.dispersionLabel.SetForegroundColour((0,0,0))
        self.qualityLabel.SetForegroundColour((0,0,0))

        fgs_camera = wx.FlexGridSizer(2, 2, 2, 15)
        fgs_camera.AddMany([(self.cameraBitDepthLabel), (self.cameraGainLabel), (self.cameraCaptureWidthLabel), (self.cameraExposureLabel) ])
        fgs_filtering = wx.FlexGridSizer(2, 2, 2, 15)
        fgs_filtering.AddMany([(self.sampleHeightLabel), (self.savgolFilterLabel), (self.runningAvgLabel), (self.stabilityLabel) ])
        fgs_calibrationStatus = wx.FlexGridSizer(2, 2, 2, 15)
        fgs_calibrationStatus.AddMany([(self.calibrateStatusLabel), (self.polyFitLabel), (self.dispersionLabel), (self.qualityLabel) ])
        fgs_miscellaneous = wx.FlexGridSizer(2, 1, 2, 15)
        fgs_miscellaneous.AddMany([ (self.saveDataLabel), (self.holdPeaksLabel)])

        hbox.AddGrowableCol(0, 1)
        hbox.AddGrowableCol(1, 2)
        hbox.AddGrowableCol(2, 2)
        hbox.AddGrowableCol(3, 2)
        hbox.AddGrowableCol(4, 2)

        hbox.Add(self.programNameLabel, proportion=0, flag=wx.ALL, border=2)
        hbox.Add(fgs_camera, proportion=0,flag=wx.ALL|wx.EXPAND, border=2)
        hbox.Add(fgs_filtering, proportion=0,flag=wx.ALL|wx.EXPAND, border=2)
        hbox.Add(fgs_calibrationStatus, proportion=0,flag=wx.ALL|wx.EXPAND, border=2)
        hbox.Add(fgs_miscellaneous, proportion=0,flag=wx.ALL|wx.EXPAND, border=2)
        self.SetSizer(hbox)
        self.bar.SetStatusText("Init done")
        
# ========================================================================================
class GraphWindow(wx.Window):
    ID_TIMER = 1

    def __init__(self, parent, id, statusbar, theSize, theName):
        sty = wx.NO_BORDER
        wx.Window.__init__(self, parent, id, theSize, style=sty)
        self.parent = parent
        
        self.SetBackgroundColour(wx.WHITE)
        self.SetCursor(wx.CROSS_CURSOR)

        # Some initalisation, just to reminds the user that a variable
        # called self.BufferBmp exists. See self.OnSize().
        self.BufferBmp = None

        # self.Bind(wx.EVT_SIZE, self.OnSize)
        self.Bind(wx.EVT_PAINT, self.OnPaint)
        # avoid flickering:
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
       
    def OnEraseBackground(self, evt):
        # do nothing, but required to avoid flickering
        pass

    def UpdateDrawing(self,evt):
        # this is called each timer-update to update the graph
        self.Refresh()
        self.Update()

    # OnPaint is executed at the app start, when resizing or when
    # the application windows becomes active.
    def OnPaint(self, event):
        dc = wx.PaintDC(self)
        #dc.BeginDrawing()
        if self.BufferBmp != None:
            dc.DrawBitmap(self.BufferBmp, 0, 0, True)

# ========================================================================================
class FWHM_resultsFrame(wx.Frame):
    def __init__(self, parent):
        wx.Frame.__init__(self, parent,  -1, title="FWHM Measurement",size=(200, 160), style=wx.FRAME_FLOAT_ON_PARENT|wx.DEFAULT_FRAME_STYLE & (~wx.CLOSE_BOX) & (~wx.MINIMIZE_BOX) & (~wx.MAXIMIZE_BOX))
        self.passActivate = None
        self.parent = parent

        theBox = wx.FlexGridSizer(6, 2, 0, 10)
        self.FWHM_Label0 = wx.StaticText(self, label="Method")
        if parent.Parent.doGaussianFWHM_Measurement:
            self.FWHM_Label0txt = wx.StaticText(self, label="Gaussian")
        else:
            self.FWHM_Label0txt = wx.StaticText(self, label="Block")
        self.FWHM_Label1 = wx.StaticText(self, label="Peak type")
        self.FWHM_Label1txt = wx.StaticText(self, label="--")
        self.FWHM_Label2 = wx.StaticText(self, label="FWHM")
        self.FWHM_Label2txt = wx.StaticText(self, label="--")
        if parent.Parent.doGaussianFWHM_Measurement:
            self.FWHM_Label3 = wx.StaticText(self, label="Peak position")
        else:
            self.FWHM_Label3 = wx.StaticText(self, label="Centre position")
        self.FWHM_Label3txt = wx.StaticText(self, label="--")
        if parent.Parent.doGaussianFWHM_Measurement:
            self.FWHM_Label4 = wx.StaticText(self, label="Peak value")
        else:
            self.FWHM_Label4 = wx.StaticText(self, label="Average value")
        self.FWHM_Label4txt = wx.StaticText(self, label="--")
        if parent.Parent.doGaussianFWHM_Measurement:
            self.FWHM_Label5 = wx.StaticText(self, label="R")
            self.FWHM_Label5txt = wx.StaticText(self, label="--")
        
        self.FWHM_Label0.SetForegroundColour((0,0,0))
        self.FWHM_Label0txt.SetForegroundColour((0,0,0))
        self.FWHM_Label1.SetForegroundColour((0,0,0))
        self.FWHM_Label1txt.SetForegroundColour((0,0,0))
        self.FWHM_Label2.SetForegroundColour((0,0,0))
        self.FWHM_Label2txt.SetForegroundColour((0,0,0))
        self.FWHM_Label3.SetForegroundColour((0,0,0))
        self.FWHM_Label3txt.SetForegroundColour((0,0,0))
        self.FWHM_Label4.SetForegroundColour((0,0,0))
        self.FWHM_Label4txt.SetForegroundColour((0,0,0))
        if parent.Parent.doGaussianFWHM_Measurement:
            self.FWHM_Label5.SetForegroundColour((0,0,0))
            self.FWHM_Label5txt.SetForegroundColour((0,0,0))

        theBox.Add(self.FWHM_Label0, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label0txt, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label1, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label1txt, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label2, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label2txt, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label3, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label3txt, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label4, proportion=0, flag=wx.ALL, border=2)
        theBox.Add(self.FWHM_Label4txt, proportion=0, flag=wx.ALL, border=2)
        if parent.Parent.doGaussianFWHM_Measurement:
            theBox.Add(self.FWHM_Label5, proportion=0, flag=wx.ALL, border=2)
            theBox.Add(self.FWHM_Label5txt, proportion=0, flag=wx.ALL, border=2)

        theBox.AddGrowableCol(0, 1)
        theBox.AddGrowableCol(1, 2)
        self.SetSizer(theBox)


        pW, pH = parent.Parent.GetSize()
        cW, cH = self.GetSize()
        self.Move(parent.Parent.GetPosition()+(pW - cW - 50, 0 + 0 + 50))
#        print(parent.Parent.GetPosition(),parent.Parent.GetPosition()+(10, 200),self.GetSize(),parent.Parent.GetSize())
#        self.Centre()
        self.Show()
#        self.Bind(wx.EVT_ACTIVATE, self.OnActivate)
        self.Bind(wx.EVT_CLOSE, self.onExit)

    def OnActivate(self, evt):
        # use self.passActivate and OnActivate() to help make user not
        # be able to access anything on this form
        if self.passActivate is not None:
            self.passActivate.Raise()

    def onExit(self, evt):
        self.parent.Parent.FWHM_DataFrameIsOpen = False
        self.Destroy()
#        pass


# ========================================================================================
class CalibrationLampSetUpFrame(wx.Frame):
    def __init__(self, parent):
        APPWIDTH = 1200
        APPHEIGHT = 200
        wx.Frame.__init__(self, parent,  -1, title="Calibration lamp set-up",size=(APPWIDTH, APPHEIGHT), style=wx.FRAME_FLOAT_ON_PARENT|wx.DEFAULT_FRAME_STYLE & (~wx.MINIMIZE_BOX) & (~wx.MAXIMIZE_BOX))
        self.passActivate = None
        self.parent = parent
        self.combined_data = []

        self.pnl = wx.Panel(self, wx.ID_ANY)
        hbox = wx.FlexGridSizer(1, 3, 0, 10)
        theDataBox = wx.FlexGridSizer(6, 2, 0, 10)
        self.FWHM_Label0 = wx.StaticText(self.pnl, label="Calibration Lamp")
        self.FWHM_Label0data = wx.StaticText(self.pnl, label=CALIBRATIONLAMPS[parent.Parent.calibrationLampId][0])

        idx = 1
        choiseList = []
        for wl in CALIBRATIONLAMPS[parent.Parent.calibrationLampId][1]:
            choiseList.append("{}: {}nm".format(idx,wl))
            idx += 1
        self.FWHM_Label1 = wx.StaticText(self.pnl, label="First peak")
#        self.FWHM_Label1data = wx.ComboBox(self, style=wx.CB_READONLY, choices=[str(wl)+"nm" for wl in CALIBRATIONLAMPS[parent.Parent.calibrationLampId][1]])
        self.FWHM_Label1data = wx.ComboBox(self.pnl, style=wx.CB_READONLY, choices=choiseList)
        self.FWHM_Label1data.SetSelection(parent.Parent.calibrationFirstPeak)
        self.Bind(wx.EVT_COMBOBOX, lambda event, parent=parent, arg="firstPeak": self.OnSelect(event, parent, arg), self.FWHM_Label1data)

        self.FWHM_Label2 = wx.StaticText(self.pnl, label="Last peak")
        self.FWHM_Label2data = wx.ComboBox(self.pnl, style=wx.CB_READONLY, choices=choiseList)
        self.FWHM_Label2data.SetSelection(parent.Parent.calibrationLastPeak)
        self.Bind(wx.EVT_COMBOBOX, lambda event, parent=parent, arg="lastPeak": self.OnSelect(event, parent, arg), self.FWHM_Label2data)

        self.FWHM_Label3 = wx.StaticText(self.pnl, label="Peak threshold (% of max)")
        self.FWHM_Label3data = wx.SpinCtrl(self.pnl, style=wx.CB_READONLY, value=str(int(parent.Parent.calibrationDetectionLevel*100)))
        self.FWHM_Label3data.SetRange(0, 100)
        self.Bind(wx.EVT_SPINCTRL, lambda event, parent=parent, arg="detectionLevel": self.OnSelect(event, parent, arg), self.FWHM_Label3data)
        
        self.FWHM_Label4 = wx.StaticText(self.pnl, label="Rejection level")
        self.FWHM_Label4data = wx.ComboBox(self.pnl, style=wx.CB_READONLY, choices=["{:.1f}nm".format(rl) for rl in np.arange (0.0,5.0,0.5)])
        self.FWHM_Label4data.SetSelection(int(parent.Parent.calibrationRejectionLevel*2))
        self.Bind(wx.EVT_COMBOBOX, lambda event, parent=parent, arg="rejectionLevel": self.OnSelect(event, parent, arg), self.FWHM_Label4data)

#        self.FWHM_Label5 = wx.StaticText(self.pnl, label="Reference accuracy (1"+chr(0x03C3)+", 68%)")
#        self.FWHM_Label5data = wx.ComboBox(self.pnl, style=wx.CB_READONLY, choices=["{:.2f}nm".format(rl) for rl in np.arange (0.0,1.0,0.05)])
#        self.FWHM_Label5data.SetSelection(int(parent.Parent.calibrationSpectrumQuality*20))
#        self.Bind(wx.EVT_COMBOBOX, lambda event, parent=parent, arg="spectrumQuality": self.OnSelect(event, parent, arg), self.FWHM_Label5data)

        self.FWHM_Label0.SetForegroundColour((0,0,0))
        self.FWHM_Label0data.SetForegroundColour((0,0,0))
        self.FWHM_Label1.SetForegroundColour((0,0,0))
        self.FWHM_Label1data.SetForegroundColour((0,0,0))
        self.FWHM_Label2.SetForegroundColour((0,0,0))
        self.FWHM_Label2data.SetForegroundColour((0,0,0))
        self.FWHM_Label3.SetForegroundColour((0,0,0))
        self.FWHM_Label3data.SetForegroundColour((0,0,0))
        self.FWHM_Label4.SetForegroundColour((0,0,0))
        self.FWHM_Label4data.SetForegroundColour((0,0,0))
#        self.FWHM_Label5.SetForegroundColour((0,0,0))
#        self.FWHM_Label5data.SetForegroundColour((0,0,0))

        theDataBox.Add(self.FWHM_Label0, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label0data, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label1, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label1data, flag=wx.TOP|wx.EXPAND, border=2)
        theDataBox.Add(self.FWHM_Label2, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label2data, flag=wx.TOP|wx.EXPAND, border=2)
        theDataBox.Add(self.FWHM_Label3, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label3data, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label4, proportion=0, flag=wx.ALL, border=2)
        theDataBox.Add(self.FWHM_Label4data, proportion=0, flag=wx.ALL, border=2)
#        theDataBox.Add(self.FWHM_Label5, proportion=0, flag=wx.ALL, border=2)
#        theDataBox.Add(self.FWHM_Label5data, proportion=0, flag=wx.ALL, border=2)
        theDataBox.AddGrowableCol(0, 1)
        theDataBox.AddGrowableCol(1, 1)

        hbox.AddGrowableCol(0, 1)
        hbox.Add(theDataBox, proportion=0,flag=wx.ALL|wx.EXPAND, border=2)

#        png = wx.Image(CALIBRATIONLAMPS[parent.Parent.calibrationLampId][6], wx.BITMAP_TYPE_ANY).ConvertToBitmap()
#        hbox.Add(wx.StaticBitmap(self, -1, png, (0, 0), (png.GetWidth(), png.GetHeight()),style=wx.EXPAND|wx.ALL))
        self.theCanvas = plot.PlotCanvas(self.pnl)
        self.theCanvas.SetMinSize((int(APPWIDTH*0.75), APPHEIGHT))
#        self.plotData(CALIBRATIONLAMPS[parent.Parent.calibrationLampId][6].replace(".png",".csv"))
        self.plotData(CALIBRATIONLAMPS[parent.Parent.calibrationLampId][6])
        hbox.Add(self.theCanvas, 1, wx.EXPAND | wx.ALL, 10)   

        self.pnl.SetSizer(hbox)

#        self.Move(parent.Parent.GetPosition()+(50, 50))
#        self.Centre()

        xp,yp = parent.Parent.GetPosition()
        wp,hp = parent.Parent.GetSize() / 2
        ws,hs = (APPWIDTH / 2, APPHEIGHT / 2)
        x,y = (int(xp+wp-ws),int(max(0,yp-30)))
        self.Move((x, y))
        self.Show()
        self.Bind(wx.EVT_CLOSE, lambda event, parent=parent: self.onExit(event, parent))
#        self.Bind(wx.EVT_CLOSE, self.onExit)
        parent.Parent.calibrationStartTime = time.time()

    def plotData(self,datafile):
        self.pnl.SetBackgroundColour(wx.WHITE)
        data = loadtxt(datafile, comments="#", delimiter=",", unpack=False)
        # Target X values for interpolation
        X_target = CALIBRATIONLAMPS[self.parent.Parent.calibrationLampId][1]
        # Extract X and Y columns from the discrete dataset
        X_discrete = data[:, 0]
        Y_discrete = data[:, 1]
        #### **Step 2: Apply Interpolation**
        interp_func = spi.interp1d(X_discrete, Y_discrete, kind='linear', fill_value="extrapolate")
        # Interpolated Y-values for X_target
        Y_target = interp_func(X_target)
        #### **Step 3: Combine Results into 2D Array**
        self.combined_data = np.column_stack((X_target, Y_target))

        markers = plot.PolyMarker(self.combined_data, legend="", colour="blue", marker="circle", size=2)
        '''

        # Annotate with Sequential Numbers
        annotations = [
            plot.PolyText([(x, y)], text=str(i+1), colour='red', size=12)
            for i, (x, y) in enumerate(combined_data)
        ]
        '''
        line = plot.PolyLine(data, legend="", colour="blue", width=1)
        titleStr = "{} theoretical profile ({} points)".format(CALIBRATIONLAMPS[self.parent.Parent.calibrationLampId][0],len(CALIBRATIONLAMPS[self.parent.Parent.calibrationLampId][1]))
#        gc = plot.PlotGraphics([markers,line], titleStr, "Wavelength [nm]", "Residual [nm]")
#        gc = plot.PlotGraphics([line], titleStr, "Wavelength [nm]", "Residual [nm]")
        gc = plot.PlotGraphics([line], titleStr, "Wavelength [nm]", "Residual [nm]")
        

        xstep = 50 # nm
        ystep = 10 # nm
        xmin = min(data[0] for data in data)
        xmax = max(data[0] for data in data)
        ymin = min(data[1] for data in data)
        ymax = max(data[1] for data in data)
        xmin = xstep*round((xmin-xstep/2)/xstep,0)
        xmax = xstep*round((xmax+xstep/2)/xstep,0)
        ymin = ystep*round((ymin-ystep/2)/ystep,0)
        ymax = ystep*round((ymax+ystep/2)/ystep,0)
        self.theCanvas.Draw(gc, xAxis=(xmin,xmax), yAxis=(ymin,ymax))

    def OnSelect(self,evt,parent,item):
        if item == "firstPeak":
            parent.Parent.calibrationFirstPeak = evt.GetSelection()
            if parent.Parent.calibrationFirstPeak >= parent.Parent.calibrationLastPeak:
                parent.Parent.calibrationLastPeak = min(len(CALIBRATIONLAMPS[parent.Parent.calibrationLampId][1])-1,parent.Parent.calibrationFirstPeak+1)
        elif item == "lastPeak":
            parent.Parent.calibrationLastPeak = evt.GetSelection()
            if parent.Parent.calibrationLastPeak <= parent.Parent.calibrationFirstPeak:
                parent.Parent.calibrationFirstPeak = max(0,parent.Parent.calibrationLastPeak-1)
        elif item == "detectionLevel":
            parent.Parent.calibrationDetectionLevel = evt.GetSelection()/100
        elif item == "rejectionLevel":
            parent.Parent.calibrationRejectionLevel = evt.GetSelection()/2
        elif item == "spectrumQuality":
            parent.Parent.calibrationSpectrumQuality = evt.GetSelection()/20
        parent.Parent.calibrationStartTime = time.time()
        self.FWHM_Label1data.SetSelection(parent.Parent.calibrationFirstPeak)
        self.FWHM_Label2data.SetSelection(parent.Parent.calibrationLastPeak)
    
    def OnPaint(self, event):
        """Custom method to annotate markers"""
        print("Annotating...")
        dc = wx.PaintDC(self)
        for i, (x, y) in enumerate(self.combined_data):
            pixel_x, pixel_y = self.WorldToClient((x, y))  # Convert world coordinates to pixel coordinates
            dc.DrawText(str(i + 1), pixel_x + 5, pixel_y - 5)  # Label the markers
    
    def onExit(self, evt, parent):
        parent.Parent.calibrationStartTime = time.time()
        self.parent.Parent.CalibrationLampSetUpFrameIsOpen = False
        self.Destroy()
#        pass

# ========================================================================================
class CalibrationResidualPlotFrame(wx.Frame):
    def __init__(self, parent):
        APPWIDTH = 600
        APPHEIGHT = 400
        self.parent = parent
        wx.Frame.__init__(self, parent,  -1, title="Calibration residuals for "+CALIBRATIONLAMPS[self.parent.Parent.calibrationLampId][0]+" light bulb",size=(APPWIDTH, APPHEIGHT), style=wx.FRAME_FLOAT_ON_PARENT|wx.DEFAULT_FRAME_STYLE & (~wx.MINIMIZE_BOX) & (~wx.MAXIMIZE_BOX))
        self.passActivate = None

        self.pnl = wx.Panel(self, -1)        
        self.theCanvas = plot.PlotCanvas(self.pnl)

        self.plotData(parent.Parent.calibrationResidualPlotDataLinear,parent.Parent.calibrationResidualPlotDataLinear,parent.Parent.calibrationResidualPlotData,parent.Parent.calibrationPolynomeDifferenceWithLinear)
        
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        mainSizer.Add(self.theCanvas, 1, wx.EXPAND | wx.ALL, 10)   
        self.pnl.SetSizer(mainSizer)
        
        xp,yp = parent.Parent.GetPosition()
        wp,hp = parent.Parent.GetSize() / 2
        ws,hs = (APPWIDTH / 2, APPHEIGHT / 2)
        x,y = (int(xp+wp/2),int(yp+100))
        self.Move((x, y))
        self.Show()
        self.Bind(wx.EVT_CLOSE, lambda event, parent=parent: self.onExit(event, parent))
        
    def plotData(self,dataLinearRaw,dataLinear,data,polydiffdata):
        self.pnl.SetBackgroundColour(wx.WHITE)
        numRejectedPoints = len(dataLinearRaw)-len(dataLinear)
        markers = plot.PolyMarker(data, legend="Corrected ({})".format(len(data)), colour="blue", marker="circle", size=2)
        markersLinear = plot.PolyMarker(dataLinear, legend="Raw, filtered ({})".format(len(dataLinear)), colour="red", marker="circle", size=1.5)
        markersLinearRaw = plot.PolyMarker(dataLinearRaw, legend="Raw, rejected ({})".format(numRejectedPoints), colour="grey", marker="circle", size=1.5)
#        line = plot.PolyLine(data, legend="", colour="blue", width=1)
        line = plot.PolyLine(polydiffdata, legend="Best-fit", colour="red", width=1)
        if self.parent.Parent.calibrationRequested:
            titleStr = POLYFITS[self.parent.Parent.polyfitDegree-1]+" Residuals (\N{GREEK SMALL LETTER CHI}"+chr(0xB2)+"={:.3f})".format(self.parent.Parent.chiSquaredPdoF)
        else:
            titleStr = POLYFITS[self.parent.Parent.polyfitDegree-1]+" Residuals (R"+chr(0xB2)+"={:.7f})".format(self.parent.Parent.R_sq)
#        gc = plot.PlotGraphics([markers,line], titleStr, "Wavelength [nm]", "Residual [nm]")

        gc = plot.PlotGraphics([markers,markersLinearRaw,markersLinear,line], titleStr, "Wavelength [nm]", "Residual [nm]")
        xstep = 50 # nm
        ystep = 2 # nm
        xmin = min(data[0] for data in dataLinearRaw)
        xmax = max(data[0] for data in dataLinearRaw)
        ymin = min(data[1] for data in dataLinearRaw)
        ymax = max(data[1] for data in dataLinearRaw)
        xmin = xstep*round((xmin-xstep/2)/xstep,0)
        xmax = xstep*round((xmax+xstep/2)/xstep,0)
        ymin = ystep*round((ymin-ystep/2)/ystep,0)
        ymax = ystep*round((ymax+ystep/2)/ystep,0)
        self.theCanvas.Draw(gc, xAxis=(xmin,xmax), yAxis=(ymin,ymax))
        self.theCanvas.enableLegend = True

    def onExit(self, evt, parent):
        self.parent.Parent.ToggleResidualPlot(evt)
#        self.parent.Parent.calibrationResidualPlotFrameIsOpen = False
        self.Destroy()

# ========================================================================================
class CameraPreviewFrame(wx.Frame):
    def __init__(self, parent):
        self.APPWIDTH = 300
        self.APPHEIGHT = 200
        self.parent = parent
        wx.Frame.__init__(self, parent,  -1, title="Full Frame Preview",size=(self.APPWIDTH, self.APPHEIGHT), style=wx.FRAME_FLOAT_ON_PARENT|wx.DEFAULT_FRAME_STYLE & (~wx.MINIMIZE_BOX) & (~wx.MAXIMIZE_BOX))
        self.passActivate = None

        self.img = wx.Image(width=1, height=1)
        theSize = self.img.GetSize()
        self.ctrl = wx.StaticBitmap(self, bitmap=self.img.ConvertToBitmap(), size = theSize)
        self.Bind(wx.EVT_SIZE, self.on_resize)
        self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
        self.ctrl.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)

        self.Centre()
        self.Show()
        self.Bind(wx.EVT_CLOSE, lambda event, parent=parent: self.onExit(event, parent))
        

    # ------------------------------------------------------------------------------------
    def update_bitmap(self):
        w, h = self.GetSize()
        iw, ih = self.img.GetSize()
#        print(w,h,iw,ih)
        # exit this function when empty image
        if iw*ih == 0:
            return
        self.ctrl.SetBitmap(self.img.Scale(w, h).ConvertToBitmap())

    # ------------------------------------------------------------------------------------
    def on_resize(self, evt):
        return
        w, h = self.GetSize()
        iw, ih = self.img.GetSize()
        if w/h > iw/ih:
            self.SetSize(w,int(w/iw*ih))
        else:
            self.SetSize(int(h/ih*iw),h)
#        pass
        #self.update_bitmap()
        
    # ------------------------------------------------------------------------------------
    def OnEraseBackground(self, evt):
        # do nothing
        pass

    def onExit(self, evt, parent):
        self.parent.Parent.ToggleCameraPreviewFrame(evt)
#        self.parent.Parent.cameraPreviewFrameIsOpen = False
        self.Destroy()

# ========================================================================================
def main():
    
    app = wx.App()
    ex = PySpectrometer3(None)
    ex.Show()
    ex.DetectCam()
    ex.StartCam()
    ex.GetCalData()
    ex.enableToolbar()
    

    app.MainLoop()

if __name__ == '__main__':
    main()